---
name: r.describe
description: Prints terse list of category values found in a raster map layer.
keywords: [ raster, metadata ]
---

# r.describe

Prints terse list of category values found in a raster map layer.

=== "Command line"

    **r.describe**
    [**-1rndi**]
    **map**=*name*
    [**null_value**=*string*]
    [**nsteps**=*integer*]
    **format**=*name*
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.describe map=name format=plain
    ```

=== "Python (grass.script)"

    *grass.script.parse_command*("***r.describe***",
        **map**,
        **null_value**=*"\*"*,
        **nsteps**=*255*,
        **format**=*"plain"*,
        **flags**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.parse_command("r.describe", map="name", format="json")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_describe*(**map**,
        **null_value**=*"\*"*,
        **nsteps**=*255*,
        **format**=*"plain"*,
        **flags**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_describe(map="name", format="json")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **map**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of raster map  
    **null_value**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;String representing NULL value  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *\**  
    **nsteps**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of quantization steps  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *255*  
    **format**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Plain text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    **-1**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print the output one value per line  
    **-r**  
    &nbsp;&nbsp;&nbsp;&nbsp;Only print the range of the data  
    **-n**  
    &nbsp;&nbsp;&nbsp;&nbsp;Suppress reporting of any NULLs  
    **-d**  
    &nbsp;&nbsp;&nbsp;&nbsp;Use the current region  
    **-i**  
    &nbsp;&nbsp;&nbsp;&nbsp;Read floating-point map as integer  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **map** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **null_value** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;String representing NULL value  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *\**  
    **nsteps** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of quantization steps  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *255*  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Plain text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1*, *r*, *n*, *d*, *i*  
    &nbsp;&nbsp;&nbsp;&nbsp;**1**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print the output one value per line  
    &nbsp;&nbsp;&nbsp;&nbsp;**r**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Only print the range of the data  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Suppress reporting of any NULLs  
    &nbsp;&nbsp;&nbsp;&nbsp;**d**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use the current region  
    &nbsp;&nbsp;&nbsp;&nbsp;**i**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Read floating-point map as integer  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **map** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **null_value** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;String representing NULL value  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *\**  
    **nsteps** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of quantization steps  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *255*  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Plain text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1*, *r*, *n*, *d*, *i*  
    &nbsp;&nbsp;&nbsp;&nbsp;**1**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print the output one value per line  
    &nbsp;&nbsp;&nbsp;&nbsp;**r**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Only print the range of the data  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Suppress reporting of any NULLs  
    &nbsp;&nbsp;&nbsp;&nbsp;**d**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use the current region  
    &nbsp;&nbsp;&nbsp;&nbsp;**i**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Read floating-point map as integer  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

***r.describe*** prints a terse listing of category values found in a
user-specified raster map layer.

***r.describe*** ignores the current geographic region and mask, and
reads the full extent of the input raster map. This functionality is
useful if the user intends to *reclassify* or *rescale* the data, since
these functions (*[r.reclass](r.reclass.md)* and
*[r.rescale](r.rescale.md)*) also ignore the current *geographic region*
and *mask*.

The ***nv*** parameter sets the string to be used to represent `NULL`
values in the module output; the default is '\*'.

The ***nsteps*** parameter sets the number of quantisation steps to
divide into the input raster map.

## NOTES

If the user selects the **-r** flag, a range of category values found in
the raster map layer will be printed. The range is divided into three
groups: negative, positive, and zero. If negative values occur, the
minimum and maximum negative values will be printed. If positive values
occur, the minimum and maximum positive values will be printed. If zero
occurs, this will be indicated. The range report will generally run
faster than the full list (the default output).

The **-d** flag can be used to force *r.describe* to respect the current
region extents when reporting raster map categories. The default behavior
is to read the full extent of the input raster map.

If the **-1** flag is specified, the output appears with one category
value/range per line.

The **-n** flag suppresses the reporting of `NULL` values.

## EXAMPLES

The following examples are from the North Carolina sample dataset:

### Print the full list of raster map categories

=== "Command line"

    ```sh
    r.describe map=zipcodes
    ```

    Output:

    ```text
    27511 27513 27518 27529 27539 27601 27603-27608 27610
    ```

 === "Python (grass.script)"

    ```python
    import grass.script as gs

    ranges = gs.parse_command("r.describe", map="zipcodes")
    ```

    The JSON output looks like:

    ```json
    {
        "has_nulls": false,
        "ranges": [
            {
                "min": 27511,
                "max": 27511
            },
            {
                "min": 27513,
                "max": 27513
            },
            {
                "min": 27518,
                "max": 27518
            },
            {
                "min": 27529,
                "max": 27529
            },
            {
                "min": 27539,
                "max": 27539
            },
            {
                "min": 27601,
                "max": 27601
            },
            {
                "min": 27603,
                "max": 27608
            },
            {
                "min": 27610,
                "max": 27610
            }
        ]
    }
    ```

### Print the raster range only

=== "Command line"

    ```sh
    r.describe map=zipcodes -r
    ```

    Output:

    ```text
    27511 thru 27610
    ```

 === "Python (grass.script)"

    ```python
    import grass.script as gs

    gs.parse_command("r.describe", map="zipcodes", flags="r", format="json")
    ```

    The returned dictionary looks like:

    ```text
    {'has_nulls': False, 'ranges': [{'min': 27511, 'max': 27610}]}
    ```

### Print raster map categories, one category per line

=== "Command line"

    ```sh
    r.describe map=zipcodes -1
    ```

    Output:

    ```text
    27511
    27513
    27518
    27529
    27539
    27601
    27603
    27604
    27605
    27606
    27607
    27608
    27610
    ```

 === "Python (grass.script)"

    ```python
    import grass.script as gs

    gs.parse_command("r.describe", map="zipcodes", flags="1", format="json")
    ```

    The returned dictionary looks like:

    ```text
    {'has_nulls': False, 'values': [27511, 27513, 27518, 27529, 27539, 27601, 27603, 27604, 27605, 27606, 27607, 27608, 27610]}
    ```

## SEE ALSO

*[g.region](g.region.md), [r.mask](r.mask.md),
[r.reclass](r.reclass.md), [r.report](r.report.md),
[r.rescale](r.rescale.md), [r.stats](r.stats.md),
[r.univar](r.univar.md)*

## AUTHOR

Michael Shapiro, U.S. Army Construction Engineering Research Laboratory

## SOURCE CODE

Available at: [r.describe source code](https://github.com/OSGeo/grass/tree/main/raster/r.describe)
([history](https://github.com/OSGeo/grass/commits/main/raster/r.describe))  
Latest change: Thursday Oct 02 14:39:13 2025 in commit [1850489](https://github.com/OSGeo/grass/commit/1850489bf42c72d8dcfe2e4db1738441b4a7dc43)
