---
name: r.hydrobasin
description: Delineates a large number of watersheds using the Memory-Efficient Watershed Delineation (MESHED) OpenMP parallel algorithm by Cho (2025).
keywords: [ raster, hydrology, watershed delineation ]
---

# r.hydrobasin

Delineates a large number of watersheds using the Memory-Efficient Watershed Delineation (MESHED) OpenMP parallel algorithm by Cho (2025).

=== "Command line"

    **r.hydrobasin**
    [**-m**]
    **direction**=*name*
    **format**=*string*
    **outlets**=*name*
    [**layer**=*string*]
    [**column**=*name*]
    **output**=*name*
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.hydrobasin direction=name format=auto outlets=name output=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.hydrobasin***",
        **direction**,
        **format**=*"auto"*,
        **outlets**,
        **layer**=*"1"*,
        **column**=*"cat"*,
        **output**,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.hydrobasin", direction="name", format="auto", outlets="name", output="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_hydrobasin*(**direction**,
        **format**=*"auto"*,
        **outlets**,
        **layer**=*"1"*,
        **column**=*"cat"*,
        **output**,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_hydrobasin(direction="name", format="auto", outlets="name", output="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **direction**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input direction raster map  
    **format**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Format of input direction raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *auto, degree, 45degree, power2*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *auto*  
    &nbsp;&nbsp;&nbsp;&nbsp;**auto**: auto-detect direction format  
    &nbsp;&nbsp;&nbsp;&nbsp;**degree**: degrees CCW from East  
    &nbsp;&nbsp;&nbsp;&nbsp;**45degree**: degrees CCW from East divided by 45 (e.g. r.watershed)  
    &nbsp;&nbsp;&nbsp;&nbsp;**power2**: powers of 2 CW from East (e.g., r.terraflow, ArcGIS)  
    **outlets**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input outlets vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    **layer**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **column**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of attribute column for watershed IDs (using a non-default column is slower)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *cat*  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output watersheds raster map  
    **-m**  
    &nbsp;&nbsp;&nbsp;&nbsp;Use less memory  
  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **direction** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input direction raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Format of input direction raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *auto, degree, 45degree, power2*  
    &nbsp;&nbsp;&nbsp;&nbsp;**auto**: auto-detect direction format  
    &nbsp;&nbsp;&nbsp;&nbsp;**degree**: degrees CCW from East  
    &nbsp;&nbsp;&nbsp;&nbsp;**45degree**: degrees CCW from East divided by 45 (e.g. r.watershed)  
    &nbsp;&nbsp;&nbsp;&nbsp;**power2**: powers of 2 CW from East (e.g., r.terraflow, ArcGIS)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *auto*  
    **outlets** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input outlets vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of attribute column for watershed IDs (using a non-default column is slower)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *cat*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output watersheds raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *m*  
    &nbsp;&nbsp;&nbsp;&nbsp;**m**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use less memory  
  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **direction** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input direction raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Format of input direction raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *auto, degree, 45degree, power2*  
    &nbsp;&nbsp;&nbsp;&nbsp;**auto**: auto-detect direction format  
    &nbsp;&nbsp;&nbsp;&nbsp;**degree**: degrees CCW from East  
    &nbsp;&nbsp;&nbsp;&nbsp;**45degree**: degrees CCW from East divided by 45 (e.g. r.watershed)  
    &nbsp;&nbsp;&nbsp;&nbsp;**power2**: powers of 2 CW from East (e.g., r.terraflow, ArcGIS)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *auto*  
    **outlets** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input outlets vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of attribute column for watershed IDs (using a non-default column is slower)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *cat*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output watersheds raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *m*  
    &nbsp;&nbsp;&nbsp;&nbsp;**m**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use less memory  
  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.hydrobasin* delineates a large number of watersheds from a flow
direction raster map and an outlets vector map using the
Memory-Efficient Watershed Delineation (MESHED) OpenMP parallel
algorithm by Cho (2025).

## NOTES

*r.hydrobasin* uses a flow direction raster map and an outlets vector
map to delineate a large number of watersheds in parallel using OpenMP.

The module recognizes three different formats of flow directions:

![image-alt](r_hydrobasin_formats.png)

*r.watershed* can be used to create an input flow direction raster map.
It can also create watersheds, but it uses an elevation map instead of a
flow direction map, which is actually one of its outputs, and does not
take outlets as input. *r.hydrobasin* is more similar to
*r.water.outlet* and *r.stream.basins*. Both modules take an input flow
direction map from *r.watershed*, but *r.water.outlet* can delineate a
watershed for one outlet point at a time and *r.stream.basins* is a
sequential module for multiple watersheds. Unlike *r.stream.basins*,
*r.hydrobasin* can use a column for watershed IDs, but using a
non-default column is slower than using the default category (cat)
column because of database queries.

For comparisons, using an i7-1370P CPU with 64GB memory and a 30-meter
flow direction map for the entire Texas (1.8 billion cells),
*r.hydrobasin* took 1 minute 27 seconds to delineate the entire state
using 60,993 outlet cells draining away (see below how to extract
draining cells) while *r.stream.basins* 5 minutes 28 seconds, both using
the category column. However, *r.hydrobasin* with a non-category column
took 6 minutes 21 seconds because of heavy database queries.

## EXAMPLES

These examples use the North Carolina sample dataset.

Calculate flow accumulation using *r.watershed* and delineate all
watersheds from draining cells using *r.hydrobasin*:

```sh
# set computational region
g.region -ap raster=elevation

# calculate drainage directions using r.watershed
r.watershed -s elevation=elevation drainage=drain

# extract draining cells
r.mapcalc ex="dcells=if(\
        (isnull(drain[-1,-1])&&abs(drain)==3)||\
        (isnull(drain[-1,0])&&abs(drain)==2)||\
        (isnull(drain[-1,1])&&abs(drain)==1)||\
        (isnull(drain[0,-1])&&abs(drain)==4)||\
        (isnull(drain[0,1])&&abs(drain)==8)||\
        (isnull(drain[1,-1])&&abs(drain)==5)||\
        (isnull(drain[1,0])&&abs(drain)==6)||\
        (isnull(drain[1,1])&&abs(drain)==7),1,null())"
r.to.vect input=dcells type=point output=dcells

# delineate all watersheds using r.hydrobasin
r.hydrobasin dir=drain outlets=dcells output=wsheds nprocs=$(nproc)
```

![image-alt](r_hydrobasin_wsheds.png)

Perform the same analysis for 10,938 bridges in North Carolina:

```sh
# set computational region
g.region -ap raster=elev_state_500m

# calculate drainage directions using r.watershed
r.watershed -s elevation=elev_state_500m drainage=drain_state

# delineate all watersheds using r.hydrobasin
r.hydrobasin dir=drain_state outlets=bridges output=bridge_wsheds nproc=$(nproc)
```

![image-alt](r_hydrobasin_bridge_wsheds.png)

## SEE ALSO

*[r.flowaccumulation](r.flowaccumulation.md),
[r.accumulate](r.accumulate.md),
[r.watershed](https://grass.osgeo.org/grass-stable/manuals/r.watershed.html),
[r.stream.extract](https://grass.osgeo.org/grass-stable/manuals/r.stream.extract.html),
[r.stream.distance](r.stream.distance.md)*

## REFERENCES

Huidae Cho, January 2025. *Avoid Backtracking and Burn Your Inputs:
CONUS-Scale Watershed Delineation Using OpenMP.* Environmental Modelling
& Software 183, 106244.
[doi:10.1016/j.envsoft.2024.106244](https://doi.org/10.1016/j.envsoft.2024.106244).

## AUTHOR

[Huidae Cho](mailto:grass4u@gmail-com), New Mexico State University

## SOURCE CODE

Available at: [r.hydrobasin source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/raster/r.hydrobasin)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/raster/r.hydrobasin))  
Latest change: Wednesday Sep 24 08:07:01 2025 in commit [239bee6](https://github.com/OSGeo/grass-addons/commit/239bee6b19866b69a97859943ff18625c28177fa)
