---
name: r.lfp
description: Calculates the longest flow paths from a flow direction raster map and a outlets vector map using the Memory-Efficient Longest Flow Path (MELFP) OpenMP parallel algorithm by Cho (2025).
keywords: [ raster, hydrology, longest flow path ]
---

# r.lfp

Calculates the longest flow paths from a flow direction raster map and a outlets vector map using the Memory-Efficient Longest Flow Path (MELFP) OpenMP parallel algorithm by Cho (2025).

=== "Command line"

    **r.lfp**
    [**-f**]
    **direction**=*name*
    **format**=*string*
    [**encoding**=*string*]
    **outlets**=*name*
    [**layer**=*string*]
    [**column**=*name*]
    [**lfp**=*name*]
    [**heads**=*name*]
    [**coordinates**=*name*]
    [**output_column**=*name*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.lfp direction=name format=auto outlets=name lfp=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.lfp***",
        **direction**,
        **format**=*"auto"*,
        **encoding**=*None*,
        **outlets**,
        **layer**=*"1"*,
        **column**=*"cat"*,
        **lfp**=*None*,
        **heads**=*None*,
        **coordinates**=*None*,
        **output_column**=*None*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.lfp", direction="name", format="auto", outlets="name", lfp="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_lfp*(**direction**,
        **format**=*"auto"*,
        **encoding**=*None*,
        **outlets**,
        **layer**=*"1"*,
        **column**=*"cat"*,
        **lfp**=*None*,
        **heads**=*None*,
        **coordinates**=*None*,
        **output_column**=*None*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_lfp(direction="name", format="auto", outlets="name", lfp="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **direction**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input flow direction raster map  
    **format**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Format of input flow direction raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *auto, degree, 45degree, power2, taudem, custom*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *auto*  
    &nbsp;&nbsp;&nbsp;&nbsp;**auto**: auto-detect direction format except taudem  
    &nbsp;&nbsp;&nbsp;&nbsp;**degree**: degrees CCW from East  
    &nbsp;&nbsp;&nbsp;&nbsp;**45degree**: degrees CCW from East divided by 45 (e.g. r.watershed)  
    &nbsp;&nbsp;&nbsp;&nbsp;**power2**: powers of 2 CW from East (e.g., r.terraflow, ArcGIS)  
    &nbsp;&nbsp;&nbsp;&nbsp;**taudem**: 1-8 for E-SE CCW, not auto-detected (e.g., TauDEM D8FlowDir)  
    &nbsp;&nbsp;&nbsp;&nbsp;**custom**: use encoding  
    **encoding**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Flow direction encoding for custom format  
    &nbsp;&nbsp;&nbsp;&nbsp;Eight integers for E,SE,S,SW,W,NW,N,NE  
    **outlets**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input outlets vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    **layer**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **column**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input attribute column for outlet IDs  
    &nbsp;&nbsp;&nbsp;&nbsp;Using a non-cat column is slower because of database access  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *cat*  
    **lfp**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output longest flow paths vector map  
    **heads**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output longest flow path heads vector map  
    **coordinates**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output longest flow path head coordinates file  
    **output_column**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output attribute column for outlet IDs  
    **-f**  
    &nbsp;&nbsp;&nbsp;&nbsp;Find full longest flow paths  
  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **direction** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input flow direction raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Format of input flow direction raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *auto, degree, 45degree, power2, taudem, custom*  
    &nbsp;&nbsp;&nbsp;&nbsp;**auto**: auto-detect direction format except taudem  
    &nbsp;&nbsp;&nbsp;&nbsp;**degree**: degrees CCW from East  
    &nbsp;&nbsp;&nbsp;&nbsp;**45degree**: degrees CCW from East divided by 45 (e.g. r.watershed)  
    &nbsp;&nbsp;&nbsp;&nbsp;**power2**: powers of 2 CW from East (e.g., r.terraflow, ArcGIS)  
    &nbsp;&nbsp;&nbsp;&nbsp;**taudem**: 1-8 for E-SE CCW, not auto-detected (e.g., TauDEM D8FlowDir)  
    &nbsp;&nbsp;&nbsp;&nbsp;**custom**: use encoding  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *auto*  
    **encoding** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Flow direction encoding for custom format  
    &nbsp;&nbsp;&nbsp;&nbsp;Eight integers for E,SE,S,SW,W,NW,N,NE  
    **outlets** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input outlets vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input attribute column for outlet IDs  
    &nbsp;&nbsp;&nbsp;&nbsp;Using a non-cat column is slower because of database access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *cat*  
    **lfp** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output longest flow paths vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, vector, *name*  
    **heads** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output longest flow path heads vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, vector, *name*  
    **coordinates** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output longest flow path head coordinates file  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    **output_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output attribute column for outlet IDs  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *f*  
    &nbsp;&nbsp;&nbsp;&nbsp;**f**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Find full longest flow paths  
  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **direction** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input flow direction raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Format of input flow direction raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *auto, degree, 45degree, power2, taudem, custom*  
    &nbsp;&nbsp;&nbsp;&nbsp;**auto**: auto-detect direction format except taudem  
    &nbsp;&nbsp;&nbsp;&nbsp;**degree**: degrees CCW from East  
    &nbsp;&nbsp;&nbsp;&nbsp;**45degree**: degrees CCW from East divided by 45 (e.g. r.watershed)  
    &nbsp;&nbsp;&nbsp;&nbsp;**power2**: powers of 2 CW from East (e.g., r.terraflow, ArcGIS)  
    &nbsp;&nbsp;&nbsp;&nbsp;**taudem**: 1-8 for E-SE CCW, not auto-detected (e.g., TauDEM D8FlowDir)  
    &nbsp;&nbsp;&nbsp;&nbsp;**custom**: use encoding  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *auto*  
    **encoding** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Flow direction encoding for custom format  
    &nbsp;&nbsp;&nbsp;&nbsp;Eight integers for E,SE,S,SW,W,NW,N,NE  
    **outlets** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input outlets vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input attribute column for outlet IDs  
    &nbsp;&nbsp;&nbsp;&nbsp;Using a non-cat column is slower because of database access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *cat*  
    **lfp** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output longest flow paths vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, vector, *name*  
    **heads** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output longest flow path heads vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, vector, *name*  
    **coordinates** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output longest flow path head coordinates file  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    **output_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output attribute column for outlet IDs  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *f*  
    &nbsp;&nbsp;&nbsp;&nbsp;**f**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Find full longest flow paths  
  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.lfp* calculates the longest flow paths for given outlet points using
the [Memory-Efficient Longest Flow Path
(MELFP)](https://github.com/HuidaeCho/melfp) OpenMP parallel algorithm
by Cho (2025).

## NOTES

*r.lfp* can automatically recognize the following three different
formats of flow directions: **degree**, **45degree**, and **power2**.
The **degree** format starts just above 0° at East (excluding 0° itself)
and goes counterclockwise up to 360°, which also corresponds to East.
The **45degree** format divides the degree format by 45°. The **power2**
format starts from 1 at East and doubles clockwise up to Northeast.

![image-alt](r_lfp_formats.png)

*r.lfp* also supports the **taudem** format, which is used by
[TauDEM](https://github.com/dtarb/TauDEM)'s D8FlowDir. This format is
not auto-detected because it shares the same encoding range of the
**45degree** format. Additionally, the module can accept any integer
encodings with the **custom** format and **encoding** option, which uses
eight numbers for E, SE, S, SW, W, NW, N, and NE. For example, to encode
the **45degree** format using this method, one can use **format=custom
encoding=8,7,6,5,4,3,2,1**.

![image-alt](r_lfp_custom_formats.png)

Unless the **-f** option is specified, *r.lfp* defaults to computing the
longest flow paths within each subwatershed, not crossing through any
outlet points. This default behavior will produce longest flow path
lines that do not overlap among subwatersheds. However, they can still
overlap within a subwatershed if they are of the same length and share
common downstream paths within that subwatershed.

With the **-f** option, the module first computes the longest flow paths
at the subwatershed level, then performs a hierarchical analysis to
derive potentially longer watershed-level flow paths, and finally
eliminates shorter paths from both the subwatershed and hierarchically
merged watershed results.

When parallel processing is enabled with the **nprocs** option, *r.lfp*
uses OpenMP's shared-memory model and the specified number of threads to
parallelize the computation per thread initially (implicit tasking
through looping) and later switch to explicit tasking for better load
balancing as threads start becoming ideal after they finish their
allocated implicit tasks. This loop-then-task approach significantly
improves computational efficiency along with highly reduced memory
usage. In its benchmark experiment, the MELFP algorithm used in this
module achieved a 66% reduction in computation time using 79% lower peak
memory with 33% higher CPU utilization, enabling faster and larger data
processing (Cho, 2025).

## EXAMPLES

These examples use the North Carolina sample dataset.

Extract all draining cells (all outlets for the elevation raster), and
calculate all watersheds and longest flow paths:

```sh
# set computational region
g.region -ap rast=elevation

# calculate drainage directions using r.watershed
r.watershed -s elev=elevation drain=drain

# extract draining cells
r.mapcalc ex="dcells=if(\
        (isnull(drain[-1,-1])&&abs(drain)==3)||\
        (isnull(drain[-1,0])&&abs(drain)==2)||\
        (isnull(drain[-1,1])&&abs(drain)==1)||\
        (isnull(drain[0,-1])&&abs(drain)==4)||\
        (isnull(drain[0,1])&&abs(drain)==8)||\
        (isnull(drain[1,-1])&&abs(drain)==5)||\
        (isnull(drain[1,0])&&abs(drain)==6)||\
        (isnull(drain[1,1])&&abs(drain)==7),1,null())"
r.to.vect input=dcells type=point output=dcells

# delineate all watersheds using r.hydrobasin
r.hydrobasin dir=drain outlets=dcells output=wsheds nproc=$(nproc)

# calculate all longest flow paths
r.lfp dir=drain outlets=dcells lfp=lfp ocol=outlet_cat nproc=$(nproc)

# or using a custom format for r.watershed drainage (8-1 for E-NE CW)
r.lfp dir=drain format=custom encoding=8,7,6,5,4,3,2,1 outlets=dcells lfp=lfp2 ocol=outlet_cat nproc=$(nproc)
```

![image-alt](r_lfp_elevation_example.png)

Perform the same analysis using the statewide DEM, elev_state_500m:

```sh
# set computational region
g.region -ap rast=elev_state_500m

# calculate drainage directions using r.watershed
r.watershed -s elev=elev_state_500m drain=nc_drain

# extract draining cells
r.mapcalc ex="nc_dcells=if(\
        (isnull(nc_drain[-1,-1])&&abs(nc_drain)==3)||\
        (isnull(nc_drain[-1,0])&&abs(nc_drain)==2)||\
        (isnull(nc_drain[-1,1])&&abs(nc_drain)==1)||\
        (isnull(nc_drain[0,-1])&&abs(nc_drain)==4)||\
        (isnull(nc_drain[0,1])&&abs(nc_drain)==8)||\
        (isnull(nc_drain[1,-1])&&abs(nc_drain)==5)||\
        (isnull(nc_drain[1,0])&&abs(nc_drain)==6)||\
        (isnull(nc_drain[1,1])&&abs(nc_drain)==7),1,null())"
r.to.vect input=nc_dcells type=point output=nc_dcells

# delineate all watersheds using r.hydrobasin
r.hydrobasin dir=nc_drain outlets=nc_dcells output=nc_wsheds nproc=$(nproc)

# calculate all longest flow paths
r.lfp dir=nc_drain outlets=nc_dcells lfp=nc_lfp ocol=outlet_cat nproc=$(nproc)

# or using a custom format for r.watershed drainage (8-1 for E-NE CW)
r.lfp dir=nc_drain format=custom encoding=8,7,6,5,4,3,2,1 outlets=nc_dcells lfp=nc_lfp2 ocol=outlet_cat nproc=$(nproc)
```

![image-alt](r_lfp_elev_state_500m_example.png)

## SEE ALSO

*[r.hydrobasin](r.hydrobasin.md),
[r.flowaccumulation](r.flowaccumulation.md),
[r.accumulate](r.accumulate.md),
[r.watershed](https://grass.osgeo.org/grass-stable/manuals/r.watershed.html)*

## REFERENCES

Huidae Cho, September 2025. *Loop Then Task: Hybridizing OpenMP
Parallelism to Improve Load Balancing and Memory Efficiency in
Continental-Scale Longest Flow Path Computation.* Environmental
Modelling & Software 193, 106630.
[doi:10.1016/j.envsoft.2025.106630](https://doi.org/10.1016/j.envsoft.2025.106630)

## AUTHOR

[Huidae Cho](mailto:grass4u@gmail-com), New Mexico State University

## SOURCE CODE

Available at: [r.lfp source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/raster/r.lfp)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/raster/r.lfp))  
Latest change: Wednesday Sep 24 07:09:59 2025 in commit [ff691fe](https://github.com/OSGeo/grass-addons/commit/ff691fe3c28aaf70f0e1c8c8a2d3beb5d55f2d6e)
