---
name: r.profile
description: Outputs the raster map layer values lying on user-defined line(s).
keywords: [ raster, profile, transect ]
---

# r.profile

Outputs the raster map layer values lying on user-defined line(s).

=== "Command line"

    **r.profile**
    [**-gc**]
    **input**=*name*
    [**output**=*name*]
    [**coordinates**=*east,north* [,*east,north*,...]]
    [**file**=*name*]
    [**resolution**=*float*]
    [**null_value**=*string*]
    [**units**=*string*]
    **format**=*name*
    [**color_format**=*name*]
    [**separator**=*character*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.profile input=name format=plain
    ```

=== "Python (grass.script)"

    *grass.script.parse_command*("***r.profile***",
        **input**,
        **output**=*"-"*,
        **coordinates**=*None*,
        **file**=*None*,
        **resolution**=*None*,
        **null_value**=*"\*"*,
        **units**=*None*,
        **format**=*"plain"*,
        **color_format**=*None*,
        **separator**=*"comma"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.parse_command("r.profile", input="name", format="json")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_profile*(**input**,
        **output**=*"-"*,
        **coordinates**=*None*,
        **file**=*None*,
        **resolution**=*None*,
        **null_value**=*"\*"*,
        **units**=*None*,
        **format**=*"plain"*,
        **color_format**=*None*,
        **separator**=*"comma"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_profile(input="name", format="json")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    **output**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of file for output (use output=- for stdout)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-*  
    **coordinates**=*east,north* [,*east,north*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Profile coordinate pairs  
    **file**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input file containing coordinate pairs  
    &nbsp;&nbsp;&nbsp;&nbsp;Use instead of the 'coordinates' option. "-" reads from stdin.  
    **resolution**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Resolution along profile (default = current region resolution)  
    **null_value**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;String representing NULL value  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *\**  
    **units**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Units  
    &nbsp;&nbsp;&nbsp;&nbsp;If units are not specified, current project units are used. Meters are used by default in geographic (latlon) projects.  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *meters, kilometers, feet, miles*  
    **format**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, csv, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Human readable text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**csv**: CSV (Comma Separated Values)  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    **color_format**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Color format  
    &nbsp;&nbsp;&nbsp;&nbsp;Color format for output values.  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *rgb, hex, hsv, triplet*  
    &nbsp;&nbsp;&nbsp;&nbsp;**rgb**: output color in RGB format  
    &nbsp;&nbsp;&nbsp;&nbsp;**hex**: output color in HEX format  
    &nbsp;&nbsp;&nbsp;&nbsp;**hsv**: output color in HSV format (experimental)  
    &nbsp;&nbsp;&nbsp;&nbsp;**triplet**: output color in colon-separated RGB format  
    **separator**=*character*  
    &nbsp;&nbsp;&nbsp;&nbsp;Field separator  
    &nbsp;&nbsp;&nbsp;&nbsp;Special characters: pipe, comma, space, tab, newline  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *comma*  
    **-g**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output easting and northing in first two columns of four column output  
    **-c**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output color values for each profile point (format controlled by color_format option; default is 'triplet' for plain output, 'hex' for JSON)  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of file for output (use output=- for stdout)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-*  
    **coordinates** : list[tuple[float, float]] | tuple[float, float] | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Profile coordinate pairs  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, coords, *east,north*  
    **file** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input file containing coordinate pairs  
    &nbsp;&nbsp;&nbsp;&nbsp;Use instead of the 'coordinates' option. "-" reads from stdin.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **resolution** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Resolution along profile (default = current region resolution)  
    **null_value** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;String representing NULL value  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *\**  
    **units** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Units  
    &nbsp;&nbsp;&nbsp;&nbsp;If units are not specified, current project units are used. Meters are used by default in geographic (latlon) projects.  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *meters, kilometers, feet, miles*  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, csv, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Human readable text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**csv**: CSV (Comma Separated Values)  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    **color_format** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Color format  
    &nbsp;&nbsp;&nbsp;&nbsp;Color format for output values.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *rgb, hex, hsv, triplet*  
    &nbsp;&nbsp;&nbsp;&nbsp;**rgb**: output color in RGB format  
    &nbsp;&nbsp;&nbsp;&nbsp;**hex**: output color in HEX format  
    &nbsp;&nbsp;&nbsp;&nbsp;**hsv**: output color in HSV format (experimental)  
    &nbsp;&nbsp;&nbsp;&nbsp;**triplet**: output color in colon-separated RGB format  
    **separator** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Field separator  
    &nbsp;&nbsp;&nbsp;&nbsp;Special characters: pipe, comma, space, tab, newline  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, separator, *character*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *comma*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *g*, *c*  
    &nbsp;&nbsp;&nbsp;&nbsp;**g**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Output easting and northing in first two columns of four column output  
    &nbsp;&nbsp;&nbsp;&nbsp;**c**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Output color values for each profile point (format controlled by color_format option; default is 'triplet' for plain output, 'hex' for JSON)  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of file for output (use output=- for stdout)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-*  
    **coordinates** : list[tuple[float, float]] | tuple[float, float] | list[float] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Profile coordinate pairs  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, coords, *east,north*  
    **file** : str | io.StringIO, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input file containing coordinate pairs  
    &nbsp;&nbsp;&nbsp;&nbsp;Use instead of the 'coordinates' option. "-" reads from stdin.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **resolution** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Resolution along profile (default = current region resolution)  
    **null_value** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;String representing NULL value  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *\**  
    **units** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Units  
    &nbsp;&nbsp;&nbsp;&nbsp;If units are not specified, current project units are used. Meters are used by default in geographic (latlon) projects.  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *meters, kilometers, feet, miles*  
    **format** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output format  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *plain, csv, json*  
    &nbsp;&nbsp;&nbsp;&nbsp;**plain**: Human readable text output  
    &nbsp;&nbsp;&nbsp;&nbsp;**csv**: CSV (Comma Separated Values)  
    &nbsp;&nbsp;&nbsp;&nbsp;**json**: JSON (JavaScript Object Notation)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *plain*  
    **color_format** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Color format  
    &nbsp;&nbsp;&nbsp;&nbsp;Color format for output values.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *rgb, hex, hsv, triplet*  
    &nbsp;&nbsp;&nbsp;&nbsp;**rgb**: output color in RGB format  
    &nbsp;&nbsp;&nbsp;&nbsp;**hex**: output color in HEX format  
    &nbsp;&nbsp;&nbsp;&nbsp;**hsv**: output color in HSV format (experimental)  
    &nbsp;&nbsp;&nbsp;&nbsp;**triplet**: output color in colon-separated RGB format  
    **separator** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Field separator  
    &nbsp;&nbsp;&nbsp;&nbsp;Special characters: pipe, comma, space, tab, newline  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, separator, *character*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *comma*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *g*, *c*  
    &nbsp;&nbsp;&nbsp;&nbsp;**g**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Output easting and northing in first two columns of four column output  
    &nbsp;&nbsp;&nbsp;&nbsp;**c**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Output color values for each profile point (format controlled by color_format option; default is 'triplet' for plain output, 'hex' for JSON)  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

This program outputs two or four column (with **-g**) data to stdout or
an ASCII file. The default two column output consists of cumulative
profile length and raster value. The optional four column output
consists of easting, northing, cumulative profile length, and raster
value. Profile end or "turning" points can be set manually with the
**coordinates** argument. The profile resolution, or distance between
profile points, is obtained from the current region resolution, or can
be manually set with the **resolution** argument.

The **coordinates** parameter can be set to comma separated geographic
coordinates for profile line endpoints. Alternatively the coordinate
pairs can be piped from the text file specified by **file** option, or
if set to "-", from `stdin`. In these cases the coordinate pairs should
be given one comma separated pair per line.

The **resolution** parameter sets the distance between each profile
point (resolution). The resolution must be provided in GRASS database
units (i.e. decimal degrees for Lat Long databases and meters for UTM).
By default *r.profile* uses the resolution of the current GRASS region.

The **null** parameter can optionally be set to change the character
string representing null values.

The optional color output (with **-c**) provides the associated GRASS colour
value for each profile point. The format of the color output is controlled by
the **color_format** option, which can be set to hex, triplet, rgb, or hsv
color formats. The default color format is triplet for plain output, and hex
for JSON output.

## NOTES

The profile resolution is measured exactly from the supplied end or
"turning" point along the profile. The end of a profile segment will be
an exact multiple of the profile resolution and will therefore not
always match the end point coordinates entered for the segmanet.

To extract the numbers in scripts, following parameters can be used:

```sh
r.profile input=dgm12.5 coordinates=3570631,5763556 2>/dev/null
```

This filters out everything except the numbers.

Option **units** enables to set units of the profile length output. If
the units are not specified, current coordinate reference system's units
will be used. In case of geographic CRS (latitude/longitude), meters are
used as default unit.

## EXAMPLES

### Extraction of values along profile defined by coordinates (variant 1)

Extract a profile with coordinates (waypoints) provided on the command
line (North Carolina data set):

```sh
g.region raster=elevation -p
r.profile -g input=elevation output=profile_points.csv \
          coordinates=641712,226095,641546,224138,641546,222048,641049,221186
```

This will extract a profile along the track defined by the three
coordinate pairs. The output file "profile_points.csv" contains
east,north,distance,value (here: elevation).

### Extraction of values along profile defined by coordinates (variant 2)

Coordinate pairs can also be "piped" into *r.profile* (variant 2a):

```sh
r.profile elevation resolution=1000 file=- << EOF
641712,226095
641546,224138
641546,222048
641049,221186
EOF
```

Coordinate pairs can also be "piped" into *r.profile* (variant 2b):

```sh
echo "641712,226095
641546,224138
641546,222048
641049,221186" > coors.txt
cat coors.txt | r.profile elevation resolution=1000 file=-
```

The output is printed into the terminal (unless the *output* parameter
is used) and looks as follows:

```sh
Using resolution: 1000 [meters]
Output columns:
Along track dist. [meters], Elevation
Approx. transect length: 1964.027749 [meters]
 0.000000 84.661507
 1000.000000 98.179062
Approx. transect length: 2090.000000 [meters]
 1964.027749 83.638138
 2964.027749 89.141029
 3964.027749 78.497757
Approx. transect length: 995.014070 [meters]
 4054.027749 73.988029
```

### JSON Output

```sh
r.profile -g input=elevation coordinates=641712,226095,641546,224138,641546,222048,641049,221186 -c format=json resolution=1000
```

The output looks as follows:

```json
[
    {
        "easting": 641712,
        "northing": 226095,
        "distance": 0,
        "value": 84.661506652832031,
        "color": "#71FF00"
    },
    {
        "easting": 641627.47980925441,
        "northing": 225098.57823319823,
        "distance": 1000.0000000000125,
        "value": 98.179061889648438,
        "color": "#FFF100"
    },
    {
        "easting": 641546,
        "northing": 224138,
        "distance": 1964.0277492948007,
        "value": 83.638137817382812,
        "color": "#64FF00"
    },
    {
        "easting": 641546,
        "northing": 223138,
        "distance": 2964.0277492948007,
        "value": 89.141029357910156,
        "color": "#A9FF00"
    },
    {
        "easting": 641546,
        "northing": 222138,
        "distance": 3964.0277492948007,
        "value": 78.497756958007812,
        "color": "#23FF00"
    },
    {
        "easting": 641546,
        "northing": 222048,
        "distance": 4054.0277492948007,
        "value": 73.988029479980469,
        "color": "#00F911"
    }
]
```

### Using JSON output with Python for plotting data

The JSON output makes for ease of integration with popular python data
science libraries. For instance, here is an example of creating a
scatterplot of distance vs elevation with color coding.

```python
import grass.script as gs
import pandas as pd
import matplotlib.pyplot as plt

# Run r.profile command
elevation = gs.read_command(
    "r.profile",
    input="elevation",
    coordinates="641712,226095,641546,224138,641546,222048,641049,221186",
    format="json",
    flags="gc",
)

# Load the JSON data into a dataframe
df = pd.read_json(elevation)

# Create the scatter plot
plt.figure(figsize=(10, 6))
plt.scatter(df["distance"], df["value"], c=df["color"], marker="o")
plt.title("Profile of Distance vs. Elevation with Color Coding")
plt.xlabel("Distance (meters)")
plt.ylabel("Elevation")
plt.grid(True)
plt.show()
```

![Profile of Distance Plot](rprofile_plot.png)  
*Profile of Distance Plot.*

### Using output in other programs

The multi column output from *r.profile* is intended for easy use in
other programs. The output can be piped (\|) directly into other
programs or saved to a file for later use. Output with geographic
coordinates (*-g*) is compatible with *[v.in.ascii](v.in.ascii.md)* and
can be piped directly into this program.

```sh
r.profile -g input=elevation coordinates=... | v.in.ascii output=elevation_profile separator=space
```

## SEE ALSO

*[v.in.ascii](v.in.ascii.md), [r.what](r.what.md),
[r.transect](r.transect.md), [wxGUI profile tool](wxGUI.md)*

## AUTHOR

[Bob Covill](mailto:bcovill@tekmap.ns.ca)

## SOURCE CODE

Available at: [r.profile source code](https://github.com/OSGeo/grass/tree/main/raster/r.profile)
([history](https://github.com/OSGeo/grass/commits/main/raster/r.profile))  
Latest change: Thursday Oct 02 14:39:13 2025 in commit [1850489](https://github.com/OSGeo/grass/commit/1850489bf42c72d8dcfe2e4db1738441b4a7dc43)
