---
name: r.stream.distance
description: Calculates distance to and elevation above streams and outlet. The module can work in stream mode where target are streams and outlets mode where targets are outlets.
keywords: [ raster, hydrology, stream network, watercourse distance ]
---

# r.stream.distance

Calculates distance to and elevation above streams and outlet.

The module can work in stream mode where target are streams and outlets mode where targets are outlets.

=== "Command line"

    **r.stream.distance**
    [**-osnm**]
    **stream_rast**=*name*
    **direction**=*name*
    [**elevation**=*name*]
    **method**=*string*
    [**distance**=*name*]
    [**difference**=*name*]
    [**memory**=*integer*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.stream.distance stream_rast=name direction=name method=downstream
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.stream.distance***",
        **stream_rast**,
        **direction**,
        **elevation**=*None*,
        **method**=*"downstream"*,
        **distance**=*None*,
        **difference**=*None*,
        **memory**=*300*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.stream.distance", stream_rast="name", direction="name", method="downstream")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_stream_distance*(**stream_rast**,
        **direction**,
        **elevation**=*None*,
        **method**=*"downstream"*,
        **distance**=*None*,
        **difference**=*None*,
        **memory**=*300*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_stream_distance(stream_rast="name", direction="name", method="downstream")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **stream_rast**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with stream network (in outlet mode is the name for outlet raster map)  
    **direction**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with flow direction  
    **elevation**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input elevation raster map  
    **method**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Calculation method  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *upstream, downstream*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *downstream*  
    **distance**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output distance raster map  
    **difference**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output elevation difference raster map  
    **memory**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Max memory used in memory swap mode (MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **-o**  
    &nbsp;&nbsp;&nbsp;&nbsp;Calculate parameters for outlets (outlet mode) instead of (default) streams  
    **-s**  
    &nbsp;&nbsp;&nbsp;&nbsp;Calculate parameters for subbasins (ignored in stream mode)  
    **-n**  
    &nbsp;&nbsp;&nbsp;&nbsp;Calculate nearest local maximum (ignored in downstream calculation)  
    **-m**  
    &nbsp;&nbsp;&nbsp;&nbsp;Use memory swap (operation is slow)  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **stream_rast** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with stream network (in outlet mode is the name for outlet raster map)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **direction** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with flow direction  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **elevation** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input elevation raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **method** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Calculation method  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *upstream, downstream*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *downstream*  
    **distance** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output distance raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **difference** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output elevation difference raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Max memory used in memory swap mode (MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *o*, *s*, *n*, *m*  
    &nbsp;&nbsp;&nbsp;&nbsp;**o**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Calculate parameters for outlets (outlet mode) instead of (default) streams  
    &nbsp;&nbsp;&nbsp;&nbsp;**s**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Calculate parameters for subbasins (ignored in stream mode)  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Calculate nearest local maximum (ignored in downstream calculation)  
    &nbsp;&nbsp;&nbsp;&nbsp;**m**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use memory swap (operation is slow)  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **stream_rast** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with stream network (in outlet mode is the name for outlet raster map)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **direction** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with flow direction  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **elevation** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input elevation raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **method** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Calculation method  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *upstream, downstream*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *downstream*  
    **distance** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output distance raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **difference** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output elevation difference raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Max memory used in memory swap mode (MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *o*, *s*, *n*, *m*  
    &nbsp;&nbsp;&nbsp;&nbsp;**o**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Calculate parameters for outlets (outlet mode) instead of (default) streams  
    &nbsp;&nbsp;&nbsp;&nbsp;**s**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Calculate parameters for subbasins (ignored in stream mode)  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Calculate nearest local maximum (ignored in downstream calculation)  
    &nbsp;&nbsp;&nbsp;&nbsp;**m**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use memory swap (operation is slow)  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

The module *r.stream.distance* can calculate: the distance to streams;
the distance to outlet; the relative elevation above streams; the
relative elevation above the outlet. The distance and the elevation are
calculated along watercourses.

The module may calculate the distance either *downstream* or *upstream*.
The default is set on downstream. The upstream method calculates the
distance to the local maximum or divide. Opposite to downstream method,
where every cell has one and only one downstream cell, in upstream
method every cell has usually more than one upstream cell. So it is
impossible to determine interchangeable path from any cell.

The upstream method offers two alternative modes switched with *-n*
flag: the *nearest local maximum/divide* is the shortest path to the
local maximum; and the *farthest maximum/divide* is the longest path
(default option). In hydrological sense, the *nearest* option means the
shortest path which a particle of water must run from the divide to
reach a particular pixel, while the *farthest* option means the possible
longest path.

In *outlets* mode, the module can optionally be used for subbasins.

In *streams* mode (default) it calculates the distance (downstream) to
the stream network taken in input. In *outlets* mode there are some
additional possibilities. If the *subbasin* option is set off, it
calculates the distance only for the outlet (downstream). If the
*subbasin* option is set on, it calculates the distance to outlet for
every subbasin separately. The *subbasin* option acts similarly to a
subbasin mask. The module *r.stream.basins* can be used to prepare the
stream network map taken in input by *r.stream.distance*. In fact it can
be used to individuate basins and subbasins.

In lat-long locations, the module gives distances not in degrees but in
meters.

## OPTIONS

- **-o**  
    Outlets. Downstream method only. Calculate distance to or elevation
    above the outlet instead of streams. It chooses only the last outlet
    in the network ignoring nodes.
- **-s**  
    Subbasins. Downstream method only. Calculate distance to or
    elevation above stream nodes instead of streams. The distance and
    the elevation difference are relative to elementary subbasins
    instead of the whole basin.
- **-n**  
    Near. For upstream method only. Calculate distance to or elevation
    above the nearest local maximum/divide. With the default option, the
    distance/elevation is calculated to the farthest possible
    maximum/divide.
- **stream\_rast**  
    Stream network: name of input stream network map, produced using
    either *r.watershed* or *r.stream.extract*. Since stream network
    maps produced by *r.watershed* and *r.stream.extract* may slightly
    differ in detail, it is required to use both stream and direction
    maps produced by the same module. Non-stream cell values must be set
    to NULL. Alternatively, in *outlet* mode, it's the raster map of the
    outlet.
- **direction**  
    Flow direction: name of input raster map with flow direction,
    produced using either *r.watershed* or *r.stream.extract*. If
    *r.stream.extract* output map is used, it is non-NULL only where
    streams occur and NULL elsewhere. NULL (nodata) cells are ignored,
    zero and negative values are valid direction data only if they vary
    from -8 to 8 (CCW from East in steps of 45 degrees). Flow direction
    map shall be of integer type (CELL).
- **elevation**  
    Elevation: name of input elevation map. It can be of type CELL,
    FCELL or DCELL.
- **method**  
    It is possible to calculate the distance with two method:
    **downstream** from any raster cell to the nearest stream cell /
    junction cell or outlet or **upstream** from any cell upstream to
    the nearest maximum or divide.
- **difference**  
    Name of output map of elevation difference to the target (outlet,
    node, stream, divide, maximum) along watercoures. The map is of
    DCELL type.
- **distance**  
    Name of output map of distance to the target (outlet, node, stream,
    divide, maximum) along watercoures. The map is of DCELL type.

## NOTES

In *stream* mode subbasin options is omitted. Input maps must be in CELL
format (default output of *r.watershed*, *r.stream.order* and
*r.stream.extract*). The distance is calculated in meters, for flat
areas not corrected by topography. Distance correction by topography may
be done using the following *r.mapcalc* formula:

```sh
r.mapcalc expression = "dist_corrected = sqrt(distance^2 + elevation^2)"
```

The module can work only if direction map, streams map and region have
the same settings. This is checked by default. If resolutions differ,
the module informs about it and stops. Region boundary and maps boundary
may differ but it may lead to unexpected results. The elevation map is
not affected by this restriction and can have whatever resolution.

It is also required that *stream\_rast* and *direction* maps come from
the same source, e.g. both from *r.stream.extract*. If the stream
network was generated with MFD method also MFD direction map must be
used.

Probably one of the most important features of *r.stream.distance* is
the ability to calculate the distance not only for streams generated by
*r.stream.extract*, but also for any integer map, as long as the
resolution corresponds to that of *direction* map. It can be a lake,
swamp, depression and lake boundaries even divided into smaller
fragments each with its own category.

## EXAMPLE

```sh

# Set the region to match with elevation map
g.region -pa raster=elevation

# Calculate flow direction and stream network
r.watershed elevation=elevation threshold=10000 drainage=direction stream=streams

# Calculate elevation above and distance to stream network using downstream method
r.stream.distance stream_rast=streams direction=direction elevation=elevation \
  method=downstream distance=distance_stream_downstream difference=difference_stream_downstream

# Calculate elevation above and distance to stream network using upstream method
r.stream.distance stream_rast=streams direction=direction elevation=elevation \
  method=upstream distance=distance_stream_upstream difference=difference_stream_upstream

# Create outlet
echo "636645,218835" | v.in.ascii -n input=- output=outlet separator=","

# Convert outlet to raster
v.to.rast input=outlet output=outlet use=cat

# Calculate distance to and elevation above outlet
r.stream.distance -o stream_rast=outlet direction=direction elevation=elevation \
  method=downstream distance=distance2outlet difference=difference2outlet

```

## SEE ALSO

*[r.mapcalc](https://grass.osgeo.org/grass-stable/manuals/r.mapcalc.html),
[r.reclass](https://grass.osgeo.org/grass-stable/manuals/r.reclass.html),
[r.stream.channel](r.stream.channel.md),
[r.stream.extract](https://grass.osgeo.org/grass-stable/manuals/r.stream.extract.html),
[r.stream.order](r.stream.order.md),
[r.stream.segment](r.stream.segment.md),
[r.stream.slope](r.stream.slope.md), [r.stream.snap](r.stream.snap.md),
[r.stream.stats](r.stream.stats.md),
[r.watershed](https://grass.osgeo.org/grass-stable/manuals/r.watershed.html)*

See also [r.streams.\*
modules](https://grasswiki.osgeo.org/wiki/R.stream.*_modules) wiki page.

## AUTHOR

Jarek Jasiewicz, Adam Mickiewicz University, Geoecology and
Geoinformation Institute.

## SOURCE CODE

Available at: [r.stream.distance source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/raster/r.stream.distance)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/raster/r.stream.distance))  
Latest change: Friday Feb 21 10:10:05 2025 in commit [7d78fe3](https://github.com/OSGeo/grass-addons/commit/7d78fe34868674c3b6050ba1924e1c5675d155c9)
