---
name: t.rast.whatcsv.py
description: Sample a space time raster dataset at specific space-time point coordinates from a csv file and write the output to stdout
keywords: [ temporal, raster, sampling, time ]
---

# t.rast.whatcsv.py

Sample a space time raster dataset at specific space-time point coordinates from a csv file and write the output to stdout

=== "Command line"

    **t.rast.whatcsv.py**
    [**-n**]
    **csv**=*name*
    **strds**=*name*
    [**output**=*name*]
    [**where**=*sql_query*]
    [**null_value**=*string*]
    [**separator**=*character*]
    **skip**=*integer*
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    t.rast.whatcsv.py csv=name strds=name skip=0
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***t.rast.whatcsv.py***",
        **csv**,
        **strds**,
        **output**=*"-"*,
        **where**=*None*,
        **null_value**=*None*,
        **separator**=*"pipe"*,
        **skip**,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("t.rast.whatcsv.py", csv="name", strds="name", skip=0)
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.t_rast_whatcsv_py*(**csv**,
        **strds**,
        **output**=*"-"*,
        **where**=*None*,
        **null_value**=*None*,
        **separator**=*"pipe"*,
        **skip**,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.t_rast_whatcsv_py(csv="name", strds="name", skip=0)
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **csv**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for the output input csv file  
    **strds**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the input space time raster dataset  
    **output**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for the output file or "-" in case stdout should be used  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-*  
    **where**=*sql_query*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL statement without 'where' keyword used in the temporal GIS framework  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: start_time &gt; '2001-01-01 12:30:00'  
    **null_value**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;String representing NULL value  
    **separator**=*character*  
    &nbsp;&nbsp;&nbsp;&nbsp;Field separator  
    &nbsp;&nbsp;&nbsp;&nbsp;Special characters: pipe, comma, space, tab, newline  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *pipe*  
    **skip**=*integer* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of header lines to skip in the csv file  
    **-n**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output header row  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **csv** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for the output input csv file  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **strds** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the input space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, strds, *name*  
    **output** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for the output file or "-" in case stdout should be used  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-*  
    **where** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL statement without 'where' keyword used in the temporal GIS framework  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: start_time &gt; '2001-01-01 12:30:00'  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *sql_query*  
    **null_value** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;String representing NULL value  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *string*  
    **separator** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Field separator  
    &nbsp;&nbsp;&nbsp;&nbsp;Special characters: pipe, comma, space, tab, newline  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, separator, *character*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *pipe*  
    **skip** : int, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of header lines to skip in the csv file  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *n*  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Output header row  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **csv** : str | io.StringIO, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for the output input csv file  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **strds** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the input space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, strds, *name*  
    **output** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for the output file or "-" in case stdout should be used  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-*  
    **where** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL statement without 'where' keyword used in the temporal GIS framework  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: start_time &gt; '2001-01-01 12:30:00'  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *sql_query*  
    **null_value** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;String representing NULL value  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *string*  
    **separator** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Field separator  
    &nbsp;&nbsp;&nbsp;&nbsp;Special characters: pipe, comma, space, tab, newline  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, separator, *character*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *pipe*  
    **skip** : int, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of header lines to skip in the csv file  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *n*  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Output header row  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*t.rast.what* is designed to sample space time raster datasets at
specific point coordinates using
[r.what](https://grass.osgeo.org/grass-stable/manuals/r.what.html)
internally. The output of
[r.what](https://grass.osgeo.org/grass-stable/manuals/r.what.html) is
transformed to different output layouts. The output layouts can be
specified using the *layout* option.

Three layouts can be specified:

- *row* - Row order, one vector sample point value per row
- *col* - Column order, create a column for each vector sample point
    of a single time step/raster layer
- *timerow* - Time order, create a column for each time step, this
    order is the original r.what output, except that the column names
    are the time stamps

Please have a look at the example to see the supported layouts.

This module is designed to run several instances of r.what to sample
subsets of a space time raster dataset in parallel. Several intermediate
text files will be created that are merged into a single file at the end
of the processing.

Coordinates can be provided as vector map using the *points* option or
as comma separated coordinate list with the *coordinates* option.

An output file can be specified using the *output* option. Stdout will
be used if no output is specified or if the *output* option is set to
"-".

## EXAMPLES

### Data preparation

In the following examples we sample a space time raster dataset that
contains 4 raster map layers. First we create the STRDS that will be
sampled with t.rast.what.

```sh
g.region s=0 n=80 w=0 e=120 b=0 t=50 res=10

# Generate data
r.mapcalc expression="a_1 = 1" -s
r.mapcalc expression="a_2 = 2" -s
r.mapcalc expression="a_3 = 3" -s
r.mapcalc expression="a_4 = 4" -s

t.create type=strds output=A title="A test" descr="A test"

t.register -i type=raster input=A maps=a_1,a_2,a_3,a_4 \
    start='1990-01-01' increment="1 month"
```

### Example 1

The first approach uses text coordinates as input and stdout as output,
the layout is one coordinate(point per column:

```sh
t.rast.what strds=A coordinates="115,36,79,45" layout=col -n

start|end|115.0000000000;36.0000000000|79.0000000000;45.0000000000
1990-01-01 00:00:00|1990-02-01 00:00:00|1|1
1990-02-01 00:00:00|1990-03-01 00:00:00|2|2
1990-03-01 00:00:00|1990-04-01 00:00:00|3|3
1990-04-01 00:00:00|1990-05-01 00:00:00|4|4
```

### Example 2

A vector map layer can be used as input to sample the STRDS. All three
available layouts are demonstrated using the vector map for sampling.

```sh
# First create the vector map layer based on random points
v.random output=points n=3 seed=1

# Row layout using a text file as output
t.rast.what strds=A points=points output=result.txt layout=row -n

cat result.txt

115.0043586274|36.3593955783|1990-01-01 00:00:00|1990-02-01 00:00:00|1
115.0043586274|36.3593955783|1990-02-01 00:00:00|1990-03-01 00:00:00|2
115.0043586274|36.3593955783|1990-03-01 00:00:00|1990-04-01 00:00:00|3
115.0043586274|36.3593955783|1990-04-01 00:00:00|1990-05-01 00:00:00|4
79.6816763826|45.2391522853|1990-01-01 00:00:00|1990-02-01 00:00:00|1
79.6816763826|45.2391522853|1990-02-01 00:00:00|1990-03-01 00:00:00|2
79.6816763826|45.2391522853|1990-03-01 00:00:00|1990-04-01 00:00:00|3
79.6816763826|45.2391522853|1990-04-01 00:00:00|1990-05-01 00:00:00|4
97.4892579600|79.2347263950|1990-01-01 00:00:00|1990-02-01 00:00:00|1
97.4892579600|79.2347263950|1990-02-01 00:00:00|1990-03-01 00:00:00|2
97.4892579600|79.2347263950|1990-03-01 00:00:00|1990-04-01 00:00:00|3
97.4892579600|79.2347263950|1990-04-01 00:00:00|1990-05-01 00:00:00|4


# Column layout order using stdout as output
t.rast.what strds=A points=points layout=col -n

start|end|115.0043586274;36.3593955783|79.6816763826;45.2391522853|97.4892579600;79.2347263950
1990-01-01 00:00:00|1990-02-01 00:00:00|1|1|1
1990-02-01 00:00:00|1990-03-01 00:00:00|2|2|2
1990-03-01 00:00:00|1990-04-01 00:00:00|3|3|3
1990-04-01 00:00:00|1990-05-01 00:00:00|4|4|4

# Timerow layout, one time series per row
# using the where statement to select a subset of the STRDS
# and stdout as output
t.rast.what strds=A points=points \
    where="start_time >= '1990-03-01'" layout=timerow -n

x|y|1990-03-01 00:00:00;1990-04-01 00:00:00|1990-04-01 00:00:00;1990-05-01 00:00:00
115.004358627375|36.3593955782903|3|4
79.681676382576|45.2391522852909|3|4
97.4892579600048|79.2347263950131|3|4
```

## SEE ALSO

*[r.what](https://grass.osgeo.org/grass-stable/manuals/r.what.html) ,
[r.neighbors](https://grass.osgeo.org/grass-stable/manuals/r.neighbors.html),
[t.rast.aggregate.ds](https://grass.osgeo.org/grass-stable/manuals/t.rast.aggregate.ds.html),
[t.rast.extract](https://grass.osgeo.org/grass-stable/manuals/t.rast.extract.html),
[t.info](https://grass.osgeo.org/grass-stable/manuals/t.info.html),
[g.region](https://grass.osgeo.org/grass-stable/manuals/g.region.html),
[r.mask](https://grass.osgeo.org/grass-stable/manuals/r.mask.html)*

## AUTHOR

Sören Gebbert, Thünen Institute of Climate-Smart Agriculture

## SOURCE CODE

Available at: [t.rast.whatcsv source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/temporal/t.rast.whatcsv)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/temporal/t.rast.whatcsv))  
Latest change: Friday Feb 21 10:10:05 2025 in commit [7d78fe3](https://github.com/OSGeo/grass-addons/commit/7d78fe34868674c3b6050ba1924e1c5675d155c9)
