---
name: v.mapcalc.py
description: Vector map calculator.
keywords: [ vector, algebra ]
---

# v.mapcalc.py

Vector map calculator.

=== "Command line"

    **v.mapcalc.py**
    **expression**=*expression*
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    v.mapcalc.py expression=expression
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***v.mapcalc.py***",
        **expression**,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("v.mapcalc.py", expression="expression")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.v_mapcalc_py*(**expression**,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.v_mapcalc_py(expression="expression")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **expression**=*expression* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Expression to evaluate  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **expression** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Expression to evaluate  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *expression*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **expression** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Expression to evaluate  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *expression*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*v.mapcalc* performs overlay and buffer functions on vector map layers.
New vector map layers can be created which are expressions of existing
vector map layers, boolean vector operations and buffer functions.

### PROGRAM USE

The module expects its input as expression in the following form:  
  
**result = expression**  
  
This structure is similar to r.mapcalc, see
*[r.mapcalc](https://grass.osgeo.org/grass-stable/manuals/r.mapcalc.html)*.
Where **result** is the name of a vector map layer that will contain the
result of the calculation and **expression** is any valid combination of
boolean and buffer operations for existing vector map layers.  
The input is given by using the first module option *expression=* . This
option passes a **quoted** expression on the command line, for
example:  

```sh
v.mapcalc expression="A = B"
```

Where **A** is the new vector map layer that will be equal to the
existing vector map layer **B** in this case.  

```sh
v.mapcalc "A = B"
```

will give the same result.

### OPERATORS AND FUNCTIONS

The module supports the following boolean vector operations:  

| Boolean Name | Operator | Meaning             | Precedence | Correspondent Function  |
|--------------|----------|---------------------|------------|-------------------------|
| AND          | &        | Intersection       | 1          | v.overlay operator=and  |
| OR           | \|       | Union              | 1          | v.overlay operator=or   |
| DISJOINT OR  | +        | Disjoint union     | 1          | v.patch                 |
| XOR          | ^        | Symmetric difference | 1        | v.overlay operator=xor  |
| NOT          | ~        | Complement         | 1          | v.overlay operator=not  |

And vector functions:

```text
 buff_p(A, size)          Buffer the points of vector map layer A with size
 buff_l(A, size)          Buffer the lines of vector map layer A with size
 buff_a(A, size)          Buffer the areas of vector map layer A with size
```

## NOTES

As shown in the operator table above, the boolean vector operators do
not have different precedence. In default setting the expression will be
left associatively evaluated. To define specific precedence use
parentheses around these expressions, for example:  

```sh
 v.mapcalc expression="D = A & B | C"
```

Here the first intermediate result is the intersection of vector map
layers **A & B**. This intermediate vector map layer is taken to create
the union with vector map **C** to get the final result **D**. It
represents the default behaviour of left associativity.

```sh
 v.mapcalc expression="D = A & (B | C)"
```

Here the first intermediate result is taken from the parenthesized union
of vector map layers **B | C**. Afterwards the intersection of the
intermediate vector map layer and **A** will be evaluated to get the
final result vector map layer **D**.  
  
It should be noticed, that the order in which the operations are
performed does matter. Different order of operations can lead to a
different result.

## EXAMPLES

This example needed specific region setting. It should work in UTM and
LL test locations.  
First set the regions extent and create two vector maps with one random
points, respectively:

```sh
g.region s=0 n=30 w=0 e=50 b=0 t=50 res=10 res3=10 -p3

v.random --o -z output=point_1 n=1 seed=1
v.random --o -z output=point_2 n=1 seed=2
v.info point_1
v.info point_2
```

Then the vector algebra is used to create buffers around those points,
cut out a subset and apply different boolean operation on the subsets in
one statement:

```sh
v.mapcalc --o expr="buff_and = (buff_p(point_1, 30.0) ~ buff_p(point_1, 20.0)) & \
                    (buff_p(point_2, 35) ~ buff_p(point_2, 25))"
v.mapcalc --o expr="buff_or  = (buff_p(point_1, 30.0) ~ buff_p(point_1, 20.0)) | \
                    (buff_p(point_2, 35) ~ buff_p(point_2, 25))"
v.mapcalc --o expr="buff_xor = (buff_p(point_1, 30.0) ~ buff_p(point_1, 20.0)) ^ \
                    (buff_p(point_2, 35) ~ buff_p(point_2, 25))"
v.mapcalc --o expr="buff_not = (buff_p(point_1, 30.0) ~ buff_p(point_1, 20.0)) ~ \
                    (buff_p(point_2, 35) ~ buff_p(point_2, 25))"
```

## REFERENCES

The use of this module requires the following software to be installed:
[PLY(Python-Lex-Yacc)](https://www.dabeaz.com/ply/)

```sh
# Ubuntu/Debian
sudo apt-get install python-ply

# Fedora
sudo dnf install python-ply
```

## SEE ALSO

*[v.overlay](https://grass.osgeo.org/grass-stable/manuals/v.overlay.html),
[v.buffer](https://grass.osgeo.org/grass-stable/manuals/v.buffer.html),
[v.patch](https://grass.osgeo.org/grass-stable/manuals/v.patch.html),
[r.mapcalc](https://grass.osgeo.org/grass-stable/manuals/r.mapcalc.html)*

## AUTHORS

Thomas Leppelt, Soeren Gebbert, Thuenen Institut, Germany

## SOURCE CODE

Available at: [v.mapcalc source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/vector/v.mapcalc)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/vector/v.mapcalc))  
Latest change: Thursday Mar 20 21:36:57 2025 in commit [7286ecf](https://github.com/OSGeo/grass-addons/commit/7286ecf7af235bfd089fb9b1b82fb383cf95f3fc)
