---
name: v.net.path
description: Finds shortest path on vector network.
keywords: [ vector, network, shortest path ]
---

# v.net.path

Finds shortest path on vector network.

=== "Command line"

    **v.net.path**
    [**-tgs**]
    **input**=*name*
    **output**=*name*
    **arc_layer**=*string*
    **arc_type**=*string* [,*string*,...]
    **node_layer**=*string*
    [**file**=*name*]
    [**arc_column**=*string*]
    [**arc_backward_column**=*string*]
    [**node_column**=*string*]
    [**dmax**=*float*]
    [**turn_layer**=*string*]
    [**turn_cat_layer**=*string*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    v.net.path input=name output=name arc_layer=1 arc_type=line,boundary node_layer=2
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***v.net.path***",
        **input**,
        **output**,
        **arc_layer**=*"1"*,
        **arc_type**=*"line,boundary"*,
        **node_layer**=*"2"*,
        **file**=*None*,
        **arc_column**=*None*,
        **arc_backward_column**=*None*,
        **node_column**=*None*,
        **dmax**=*1000*,
        **turn_layer**=*"3"*,
        **turn_cat_layer**=*"4"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("v.net.path", input="name", output="name", arc_layer="1", arc_type="line,boundary", node_layer="2")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.v_net_path*(**input**,
        **output**,
        **arc_layer**=*"1"*,
        **arc_type**=*"line,boundary"*,
        **node_layer**=*"2"*,
        **file**=*None*,
        **arc_column**=*None*,
        **arc_backward_column**=*None*,
        **node_column**=*None*,
        **dmax**=*1000*,
        **turn_layer**=*"3"*,
        **turn_cat_layer**=*"4"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.v_net_path(input="name", output="name", arc_layer="1", arc_type="line,boundary", node_layer="2")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output vector map  
    **arc_layer**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Arc layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **arc_type**=*string* [,*string*,...] **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Arc type  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *line, boundary*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *line,boundary*  
    **node_layer**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Node layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2*  
    **file**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of file containing start and end points. If not given, read from stdin  
    **arc_column**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Arc forward/both direction(s) cost column (number)  
    **arc_backward_column**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Arc backward direction cost column (number)  
    **node_column**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Node cost column (number)  
    **dmax**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum distance to the network  
    &nbsp;&nbsp;&nbsp;&nbsp;If start/end are given as coordinates. If start/end point is outside this threshold, the path is not found and error message is printed. To speed up the process, keep this value as low as possible.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1000*  
    **turn_layer**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer with turntable  
    &nbsp;&nbsp;&nbsp;&nbsp;Relevant only with -t flag  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *3*  
    **turn_cat_layer**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer with unique categories used in turntable  
    &nbsp;&nbsp;&nbsp;&nbsp;Relevant only with -t flag  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *4*  
    **-t**  
    &nbsp;&nbsp;&nbsp;&nbsp;Use turntable  
    **-g**  
    &nbsp;&nbsp;&nbsp;&nbsp;Use geodesic calculation for longitude-latitude projects  
    **-s**  
    &nbsp;&nbsp;&nbsp;&nbsp;Write output as original input segments, not each path as one line.  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, vector, *name*  
    **arc_layer** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Arc layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **arc_type** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Arc type  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *line, boundary*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *line,boundary*  
    **node_layer** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Node layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2*  
    **file** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of file containing start and end points. If not given, read from stdin  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **arc_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Arc forward/both direction(s) cost column (number)  
    **arc_backward_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Arc backward direction cost column (number)  
    **node_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Node cost column (number)  
    **dmax** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum distance to the network  
    &nbsp;&nbsp;&nbsp;&nbsp;If start/end are given as coordinates. If start/end point is outside this threshold, the path is not found and error message is printed. To speed up the process, keep this value as low as possible.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1000*  
    **turn_layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer with turntable  
    &nbsp;&nbsp;&nbsp;&nbsp;Relevant only with -t flag  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *3*  
    **turn_cat_layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer with unique categories used in turntable  
    &nbsp;&nbsp;&nbsp;&nbsp;Relevant only with -t flag  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *4*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *t*, *g*, *s*  
    &nbsp;&nbsp;&nbsp;&nbsp;**t**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use turntable  
    &nbsp;&nbsp;&nbsp;&nbsp;**g**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use geodesic calculation for longitude-latitude projects  
    &nbsp;&nbsp;&nbsp;&nbsp;**s**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Write output as original input segments, not each path as one line.  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, vector, *name*  
    **arc_layer** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Arc layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **arc_type** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Arc type  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *line, boundary*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *line,boundary*  
    **node_layer** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Node layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2*  
    **file** : str | io.StringIO, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of file containing start and end points. If not given, read from stdin  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **arc_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Arc forward/both direction(s) cost column (number)  
    **arc_backward_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Arc backward direction cost column (number)  
    **node_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Node cost column (number)  
    **dmax** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum distance to the network  
    &nbsp;&nbsp;&nbsp;&nbsp;If start/end are given as coordinates. If start/end point is outside this threshold, the path is not found and error message is printed. To speed up the process, keep this value as low as possible.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1000*  
    **turn_layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer with turntable  
    &nbsp;&nbsp;&nbsp;&nbsp;Relevant only with -t flag  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *3*  
    **turn_cat_layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer with unique categories used in turntable  
    &nbsp;&nbsp;&nbsp;&nbsp;Relevant only with -t flag  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *4*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *t*, *g*, *s*  
    &nbsp;&nbsp;&nbsp;&nbsp;**t**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use turntable  
    &nbsp;&nbsp;&nbsp;&nbsp;**g**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use geodesic calculation for longitude-latitude projects  
    &nbsp;&nbsp;&nbsp;&nbsp;**s**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Write output as original input segments, not each path as one line.  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*v.net.path* determines least costly, e.g. shortest or fastest path(s)
on a vector network.

Costs may be either line lengths, or attributes saved in a database
table. These attribute values are taken as costs of whole segments, not
as costs to traverse a length unit (e.g. meter) of the segment. For
example, if the speed limit is 100 km / h, the cost to traverse a 10 km
long road segment must be calculated as

```sh
length / speed = 10 km / (100 km/h) = 0.1 h.
```

Supported are cost assignments for both arcs and nodes, and also
different costs for both directions of a vector line. For areas, costs
will be calculated along boundary lines.

The input vector needs to be prepared with `v.net operation=connect` in
order to connect points representing center nodes to the network.

Nodes and arcs can be closed using `cost = -1`.

Least cost paths are written to the output vector map with an attached
attribute table.

Nodes can be

- piped into the program from file or from stdin, or
- defined in the graphical user interface ("enter values
  interactively").

The syntax is as follows:

```sh
id start_point_category end_point_category
```

(Example: 1 1 2)

or

```sh
id start_point_x start_point_y end_point_x end_point_y
```

Points specified by category must be exactly on network nodes, and the
input vector map needs to be prepared with `v.net operation=connect`.

When specifying coordinates, the next network node to a given coordinate
pair is used.

The attribute table will contain the following attributes:

- `cat` - path unique category assigned by module
- `id` - path id (read from input)
- `fcat` - from point category
- `tcat` - to point category
- `sp` - result status:
  - 0 - OK, path found
  - 1 - node is not reachable
  - 2 - point of given category does not exist
- `cost` - travelling costs (on the network, not to/from network)
- `fdist` - the distance from first point to the network
- `tdist` - the distance from the network to second point

Application of flag **-t** enables a turntable support. This flag
requires additional parameters **turn_layer** and **turn_cat_layer**
that are otherwise ignored. The turntable allows to model e.g. traffic
code, where some turns may be prohibited. This means that the input
layer is expanded by turntable with costs of every possible turn on any
possible node (intersection) in both directions. Turntable can be
created by the *[v.net](v.net.md)* module. For more information about
turns in the vector network analyses see [wiki
page](https://grasswiki.osgeo.org/wiki/Turns_in_the_vector_network_analysis).

## NOTES

Nodes and arcs can be closed using `cost = -1`.

If the cost columns **arc_column**, **arc_backward_column** and
**node_column** are not specified, the length of network segments is
measured and zero costs are assumed for nodes.

When using attributes, the length of segments is not used. To get
accurate results, the line length must be taken into account when
assigning costs as attributes. For example, to get the *fastest path*,
the columns 'max_speed' and 'length' are required. The correct fastest
path can then be found by specifying `arc_column=length/max_speed`. If
not yet existing, the column containing the line length ("length") has
to added to the attributes table using *[v.to.db](v.to.db.md)*.

## EXAMPLE

Shortest (red) and fastest (blue) path between two digitized nodes
(Spearfish):

![v.net.path example](vnetpath.png)

```sh
# Spearfish

echo "1|601955.1|4916944.9|start
2|594385.6|4921565.2|end" | v.in.ascii in=- cat=1 x=2 y=3 out=startend col="cat integer, \
                         east double precision, north double precision, label varchar(6)"

v.db.select startend

g.copy vect=roads,myroads

# create lines map connecting points to network
v.net myroads points=startend out=myroads_net op=connect thresh=500 arc_layer=1 node_layer=2

# set up costs

# create unique categories for each road in layer 3
v.category in=myroads_net out=myroads_net_time opt=add cat=1 layer=3 type=line

# add new table for layer 3
v.db.addtable myroads_net_time layer=3 col="cat integer,label varchar(43),length double precision,speed double precision,cost double precision,bcost double precision"

# copy road type to layer 3
v.to.db myroads_net_time layer=3 qlayer=1 opt=query qcolumn=label columns=label

# upload road length in miles
v.to.db myroads_net_time layer=3 type=line option=length col=length unit=miles

# set speed limits in miles / hour
v.db.update myroads_net_time layer=3 col=speed val="5.0"
v.db.update myroads_net_time layer=3 col=speed val="75.0" where="label='interstate'"
v.db.update myroads_net_time layer=3 col=speed val="75.0" where="label='primary highway, hard surface'"
v.db.update myroads_net_time layer=3 col=speed val="50.0" where="label='secondary highway, hard surface'"
v.db.update myroads_net_time layer=3 col=speed val="25.0" where="label='light-duty road, improved surface'"
v.db.update myroads_net_time layer=3 col=speed val="5.0" where="label='unimproved road'"

# define traveling costs as traveling time in minutes:

# set forward costs
v.db.update myroads_net_time layer=3 col=cost val="length / speed * 60"
# set backward costs
v.db.update myroads_net_time layer=3 col=bcost val="length / speed * 60"

# ... the 'start' and 'end' nodes have category number 1 and 2

# Shortest path: ID as first number, then cat1 and cat2
echo "1 1 2" | v.net.path myroads_net_time arc_layer=3 node_layer=2 out=mypath

# Fastest path: ID as first number, then cat1 and cat2
echo "1 1 2" | v.net.path myroads_net_time arc_layer=3 node_layer=2 arc_column=cost arc_backward_column=bcost out=mypath_time
```

To display the result, run for example:

```sh
g.region vector=myroads_net
d.mon x0
d.vect myroads_net
# show shortest path
d.vect mypath col=red width=2
# show fastest path
d.vect mypath_time col=blue width=2

# start and end point
d.vect myroads_net icon=basic/triangle fcol=green size=12 layer=2
d.font font=Vera
d.vect startend disp=cat type=point lsize=14 layer=2
```

## SEE ALSO

*[d.path](d.path.md), [v.net](v.net.md), [v.net.alloc](v.net.alloc.md),
[v.net.iso](v.net.iso.md), [v.net.salesman](v.net.salesman.md),
[v.net.steiner](v.net.steiner.md), [v.to.db](v.to.db.md)*

## AUTHORS

Radim Blazek, ITC-Irst, Trento, Italy  
Documentation: Markus Neteler, Markus Metz

### TURNS SUPPORT

The turns support was implemnented as part of GRASS GIS turns cost
project at Czech Technical University in Prague, Czech Republic.

Implementation: Stepan Turek  
Documentation: Lukas Bocan, Eliska Kyzlikova, Viera Bejdova  
Mentor: Martin Landa

## SOURCE CODE

Available at: [v.net.path source code](https://github.com/OSGeo/grass/tree/main/vector/v.net.path)
([history](https://github.com/OSGeo/grass/commits/main/vector/v.net.path))  
Latest change: Friday May 09 14:33:40 2025 in commit [b356c7e](https://github.com/OSGeo/grass/commit/b356c7e108fcbeca77474c8e83ebec8f1e7b3e4a)
