---
name: v.surf.mass
description: Performs mass-preserving area interpolation.
keywords: [ vector, surface, interpolation, pycnophylactic interpolation ]
---

# v.surf.mass

Performs mass-preserving area interpolation.

=== "Command line"

    **v.surf.mass**
    [**-z**]
    **input**=*name*
    [**weight**=*name*]
    [**layer**=*string*]
    [**column**=*name*]
    **output**=*name*
    [**iterations**=*integer*]
    [**threshold**=*float*]
    [**memory**=*integer*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    v.surf.mass input=name output=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***v.surf.mass***",
        **input**,
        **weight**=*None*,
        **layer**=*"1"*,
        **column**=*None*,
        **output**,
        **iterations**=*100*,
        **threshold**=*1e-8*,
        **memory**=*300*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("v.surf.mass", input="name", output="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.v_surf_mass*(**input**,
        **weight**=*None*,
        **layer**=*"1"*,
        **column**=*None*,
        **output**,
        **iterations**=*100*,
        **threshold**=*1e-8*,
        **memory**=*300*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.v_surf_mass(input="name", output="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector area map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    **weight**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of optional weighing raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    **layer**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **column**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of attribute column with values to approximate  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    **iterations**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum number of iterations  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *100*  
    **threshold**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Threshold for iterations  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1e-8*  
    **memory**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used for raster output (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **-z**  
    &nbsp;&nbsp;&nbsp;&nbsp;Use centroid z coordinates for approximation (3D vector maps only)  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector area map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **weight** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of optional weighing raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of attribute column with values to approximate  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **iterations** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum number of iterations  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *100*  
    **threshold** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Threshold for iterations  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1e-8*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used for raster output (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *z*  
    &nbsp;&nbsp;&nbsp;&nbsp;**z**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use centroid z coordinates for approximation (3D vector maps only)  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector area map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **weight** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of optional weighing raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **layer** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number or name  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector features can have category values in different layers. This number determines which layer to use. When used with direct OGR access this is the layer name.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of attribute column with values to approximate  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dbcolumn, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **iterations** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum number of iterations  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *100*  
    **threshold** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Threshold for iterations  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1e-8*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used for raster output (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *z*  
    &nbsp;&nbsp;&nbsp;&nbsp;**z**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use centroid z coordinates for approximation (3D vector maps only)  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*v.surf.mass* creates a raster surface from vector areas, preserving the
value of the area attribute. For example, if the selected area attibute
is the population count, the sum of all pixel values in a given area is
equal to the area's population count.

## NOTES

The current region needs to be prepared with
[g.region](https://grass.osgeo.org/grass-stable/manuals/g.region.html),
choosing a resolution such that the smallest area is covered by at least
four pixels. The current region should be completely inside the bounding
box of the vector.

## EXAMPLE

### Pycnophylactic interpolation of Voronoi triangles

Pycnophylactic interpolation of Voronoi triangles using annual precipitation
in the North Carolina sample data:

```sh
# setting the region
g.region -p raster=elev_state_500m

# create Voronoi diagram based on meteorological stations
v.voronoi input=precip_30ynormals output=precip_annual

# List of attributes for the vector precip_annual
v.info -c precip_annual

# v.surf.mass converts attributes to density, but rainfall is
# typically measured in mm which is the same for all cells in the
# same input area, thus:
# new column for area size and adjusted precipitation
v.db.addcolumn map=precip_annual \
     column="area double precision, prec_adj double precision"
v.to.db map=precip_annual column=area option=area units=meters

# Getting the size of the smallest area
v.db.univar precip_annual column=area

# The smallest area with some population is 1.20789e+08 square meters
# and with a resolution of 5000 meters covered by appr. four pixels
# (depending on the shape of the area). Adjust region for that:
g.region res=5000 -ap

# adjust precipitation values: multiply by area size, dived by pixel size
v.db.update map=precip_annual column=prec_adj \
     qcolumn="annual * area / 25000000"

# mass-preserving area interpolation
v.surf.mass input=precip_annual output=precip_annual_pycno column=prec_adj iterations=200

# rasterize Voronoi diagram for comparison
v.to.rast precip_annual out=precip_annual_voronoi type=area use=attr attrcolumn=annual

# verify results
d.mon wx0
d.rast.leg precip_annual_voronoi
d.rast.leg precip_annual_pycno
```

![Annual precipitation (30 years avg.) of North Carolina shown as Voronoi
diagram](v_surf_mass_voronoi.png)  
*Annual precipitation (30 years avg.) of North Carolina shown as Voronoi
diagram based on meteorological stations (perspective view in NVIZ).*

![Smooth Pycnophylactic Interpolation of annual precipitation](v_surf_mass_pycno.png)  
*Smooth Pycnophylactic Interpolation of annual precipitation (30 years
avg.) of North Carolina using the Voronoi diagram map based on
meteorological stations (perspective view in NVIZ).*

## REFERENCES

Tobler WR. 1979. Smooth Pycnophylactic Interpolation for Geographical
Regions. Journal of the American Statistical Association, 74 (367):
519-530
([PDF](https://people.geog.ucsb.edu/~kclarke/Geography232/Pycno.pdf)).

## AUTHOR

Markus Metz

## SOURCE CODE

Available at: [v.surf.mass source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/vector/v.surf.mass)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/vector/v.surf.mass))  
Latest change: Wednesday Feb 26 00:42:43 2025 in commit [cd37bac](https://github.com/OSGeo/grass-addons/commit/cd37bac0690006b222682002e6775b317e694e68)
