---
name: r.mregression.series.py
description: Calculates multiple regression between time series: Y(t) = b1*X1(t) + ... + bn*Xn(t).
keywords: [ raster, statistics, regression ]
---

# r.mregression.series.py

Calculates multiple regression between time series: Y(t) = b1*X1(t) + ... + bn*Xn(t).

=== "Command line"

    **r.mregression.series.py**
    **samples**=*string*
    **result_prefix**=*string*
    [**model**=*string*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.mregression.series.py samples=string result_prefix=string
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.mregression.series.py***",
        **samples**,
        **result_prefix**,
        **model**=*"ols"*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.mregression.series.py", samples="string", result_prefix="string")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_mregression_series_py*(**samples**,
        **result_prefix**,
        **model**=*"ols"*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_mregression_series_py(samples="string", result_prefix="string")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **samples**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;File contains list of input and output rasters  
    **result_prefix**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Prefix for names of result raster (rasters of regression coefficients)  
    **model**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;model type: ols (ordinary least squares), rlm (robust linear model)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *ols*  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **samples** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;File contains list of input and output rasters  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as:   
    **result_prefix** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Prefix for names of result raster (rasters of regression coefficients)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as:   
    **model** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;model type: ols (ordinary least squares), rlm (robust linear model)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as:   
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *ols*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **samples** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;File contains list of input and output rasters  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as:   
    **result_prefix** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Prefix for names of result raster (rasters of regression coefficients)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as:   
    **model** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;model type: ols (ordinary least squares), rlm (robust linear model)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as:   
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *ols*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.mregression.series* is a module to calculate multiple linear
regression parameters between several time series, e.g. NDVI and
elevation, precipitation. It s a front-end for models from
*python-statmodels* package.

The module makes each output cell value a function of the values
assigned to the corresponding cells in the input raster map series.

The module assumes a simple linear regression of the form

```text
    Y(t) = b1 * X1(t) + b2 * X2(t) + ... + bn * Xn(t)
```

The module uses two models: ordinary least squares and robust linear
models.

## NOTES

The module performs multiple linear regression, use
[r.regression.series](r.regression.series.md) for regression with one
predictor.

The number of predictor variables (*X* maps) must be the same in each
(time) series (see examples below). If the different predictors have
different or irregular time intervals, NULL raster maps can be inserted
into time series to make time intervals equal.

The list of raster inputs (including NULLs) is passed to the regression
function. The function computes the parameters over the non-NULL values,
producing a NULL result only if there aren't enough non-NULL values for
computing.

## EXAMPLES

The most important paramether is *samples*; it provides the list of *Y*
and *X* maps. The parameter is the name of csv file of the next
structure: the first line is a header, other lines provide names of the
*Y* and *X* maps. The header contains the names of the input and output
variables.

For example the csv file for regression between NDVI and (elevation,
precipitation)

```text
    NDVI = b1*Elevation + b2*Precipitation
```

could be the next file:

```csv
y,elevation,precipipation
ndvi_1,elev_1,precip_1
ndvi_2,elev_2,precip_2
...
ndvi_n,elev_n,precip_n
```

"ndvi\_t" are names of the NDVI rasters, "precip\_t" are names of
precipitation rasters. The names of the first and the second predictor
variables are "elevation" and "precipitation" accordingly.

The second paramether is *result\_prefix*. It is used for construction
of the coefficient names. For example if result\_prefix="coef.", the
names of the regression coefficients will be "coef.elevation" and
"coef.precipitation".

```sh
r.mregression.series samples=settings result_prefix="coef."
```

If the regression model includes the intercept

```text
    NDVI = b0 + b1*Elevation + b2*Precipitation
```

then the constant map should be used:

```sh
r.mapcalc "ones = 1.0"
```

and the csv file is:

```csv
y,offset,elevation,precipipation
ndvi_1,ones,elev_1,precip_1
ndvi_2,ones,elev_2,precip_2
...
ndvi_n,ones,elev_n,precip_n
```

Then the command

```sh
r.mregression.series samples=settings result_prefix="coef."
```

produces three raster maps: "coef.offset", "coef.elevation",
"coef.precipitation".

### EXAMPLE 1

Create test data for the example. Suppose we have five *Y* maps and 5
pairs of predictor *X* = *(x1, x2)* maps.

Create *X* variables (random numbers):

```sh
r.mapcalc -s "x11 = rand(0, 20)"
r.mapcalc -s "x21 = rand(0, 20)"
r.mapcalc -s "x31 = rand(0, 20)"
r.mapcalc -s "x41 = rand(0, 20)"
r.mapcalc -s "x51 = rand(0, 20)"
```

```sh
r.mapcalc -s "x12 = rand(0, 20)"
r.mapcalc -s "x22 = rand(0, 20)"
r.mapcalc -s "x32 = rand(0, 20)"
r.mapcalc -s "x42 = rand(0, 20)"
r.mapcalc -s "x52 = rand(0, 20)"
```

Create constant raster for the intercept:

```sh
r.mapcalc  "ones = 1.0"
```

Suppose *Y* is a linear function of *x1* and *x2* variables plus a
random error. (For testing purposes we assume that *Y* = *12 + 5\*x1 +
3\*x2*). Create 5 Y rasters:

```sh
r.mapcalc -s "y1 = 12 + 5* x11 + 3*x12 + rand(0, 4)"
r.mapcalc -s "y2 = 12 + 5* x21 + 3*x22 + rand(0, 4)"
r.mapcalc -s "y3 = 12 + 5* x31 + 3*x32 + rand(0, 4)"
r.mapcalc -s "y4 = 12 + 5* x41 + 3*x42 + rand(0, 4)"
r.mapcalc -s "y5 = 12 + 5* x51 + 3*x52 + rand(0, 4)"
```

So we have five test rasters *Y* and *X*. Forget for a moment that we
know the function and try to find the coeffitients.

Create *samples* csv file:

```sh
echo "y,bias,x1,x2
y1,ones,x11,x12
y2,ones,x21,x22
y3,ones,x31,x32
y4,ones,x41,x42
y5,ones,x51,x52" > settings.csv
```

Run the command

```sh
r.mregression.series samples=settings.csv result_prefix="coef."
```

Three raster maps will be created: "coef.bias", "coef.x1", "coef.x2".
This rasters contains the fitted coefitients.

## SEE ALSO

*[r.regression.series](r.regression.series.md),
[r.series](https://grass.osgeo.org/grass-stable/manuals/r.series.html),
[r.regression.line](https://grass.osgeo.org/grass-stable/manuals/r.regression.line.html),
[g.list](https://grass.osgeo.org/grass-stable/manuals/g.list.html),
[g.region](https://grass.osgeo.org/grass-stable/manuals/g.region.html)*

## AUTHOR

Dmitry Kolesov

## SOURCE CODE

Available at: [r.mregression.series source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/raster/r.mregression.series)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/raster/r.mregression.series))  
Latest change: Thursday Mar 20 21:36:57 2025 in commit [7286ecf](https://github.com/OSGeo/grass-addons/commit/7286ecf7af235bfd089fb9b1b82fb383cf95f3fc)
