---
name: r.pi.corearea
description: Variable edge effects and core area analysis
keywords: [ raster, landscape structure analysis, core area analysis ]
---

# r.pi.corearea

Variable edge effects and core area analysis

=== "Command line"

    **r.pi.corearea**
    [**-a**]
    **input**=*name*
    **costmap**=*string*
    [**propmap**=*string*]
    **output**=*name*
    **keyval**=*integer*
    **buffer**=*integer*
    **distance**=*float*
    **angle**=*float*
    **stats**=*string*
    **propmethod**=*string*
    [**dist_weight**=*float*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.pi.corearea input=name costmap=string output=name keyval=0 buffer=0 distance=0.0 angle=0.0 stats=average propmethod=linear
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.pi.corearea***",
        **input**,
        **costmap**,
        **propmap**=*None*,
        **output**,
        **keyval**,
        **buffer**,
        **distance**,
        **angle**,
        **stats**,
        **propmethod**,
        **dist_weight**=*None*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.pi.corearea", input="name", costmap="string", output="name", keyval=0, buffer=0, distance=0.0, angle=0.0, stats="average", propmethod="linear")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_pi_corearea*(**input**,
        **costmap**,
        **propmap**=*None*,
        **output**,
        **keyval**,
        **buffer**,
        **distance**,
        **angle**,
        **stats**,
        **propmethod**,
        **dist_weight**=*None*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_pi_corearea(input="name", costmap="string", output="name", keyval=0, buffer=0, distance=0.0, angle=0.0, stats="average", propmethod="linear")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    **costmap**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the cost map raster file  
    **propmap**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the propagation cost map raster file  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    **keyval**=*integer* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Key value  
    **buffer**=*integer* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Buffer size  
    **distance**=*float* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Cone of effect radius  
    **angle**=*float* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Cone of effect angle  
    **stats**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Statistical method to perform on the values  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *average, median*  
    **propmethod**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Propagation method  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *linear, exponential*  
    **dist_weight**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Parameter for distance weighting. &lt;0.5 - rapid decrease; 0.5 - linear decrease; &gt; 0.5 - slow decrease; 1 - no decrease  
    **-a**  
    &nbsp;&nbsp;&nbsp;&nbsp;Set for 8 cell-neighbors. 4 cell-neighbors are default  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **costmap** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the cost map raster file  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **propmap** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the propagation cost map raster file  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **keyval** : int, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Key value  
    **buffer** : int, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Buffer size  
    **distance** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Cone of effect radius  
    **angle** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Cone of effect angle  
    **stats** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Statistical method to perform on the values  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *average, median*  
    **propmethod** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Propagation method  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *linear, exponential*  
    **dist_weight** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Parameter for distance weighting. &lt;0.5 - rapid decrease; 0.5 - linear decrease; &gt; 0.5 - slow decrease; 1 - no decrease  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *a*  
    &nbsp;&nbsp;&nbsp;&nbsp;**a**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Set for 8 cell-neighbors. 4 cell-neighbors are default  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **costmap** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the cost map raster file  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **propmap** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the propagation cost map raster file  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **keyval** : int, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Key value  
    **buffer** : int, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Buffer size  
    **distance** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Cone of effect radius  
    **angle** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Cone of effect angle  
    **stats** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Statistical method to perform on the values  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *average, median*  
    **propmethod** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Propagation method  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *linear, exponential*  
    **dist_weight** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Parameter for distance weighting. &lt;0.5 - rapid decrease; 0.5 - linear decrease; &gt; 0.5 - slow decrease; 1 - no decrease  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *a*  
    &nbsp;&nbsp;&nbsp;&nbsp;**a**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Set for 8 cell-neighbors. 4 cell-neighbors are default  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

Edge effects and core area analysis of landcover fragments. This module
can compute static edge effects (defined edge depth) and dynamic edge
effects (based on surrounding landscape). The impact of the surrounding
landscape can be accounted for and the resulting core area is provided.

## NOTES

This module is generating core areas based on defined edge depths. The
edge depths can be increased by the values of a *costmap* (e.g. urban
areas could have a more severe impact than secondary forest on forest
fragments). Moreover a friction map ( *propmap* within the fragments can
lower the impact of surrounding landcover types and hence an increased
edge depth (e.g. a river or escarpment which might lower the edge
effects). Moreover a *dist\_weight* can be assigned in order to increase
the weight of closer pixel values.

### Distance weight

The assigned distance weight is computed as:  
w(d) = 1 - (d / d\_max)^(tan(dist\_weight \* 0.5 \* pi))  
where:  

- d = Distance of the respective cell
- d\_max - the defined maximum distance
- dist\_weight - the parameter how to weight the pixel values in the
    landscape depending on the distance  

the *dist\_weight* has a range between 0 and 1 and results in:

- 0 \< dist\_weight \< 0.5: the weighting curve decreases at low
    distances to the fragment and lowers to a weight of 0 at d=d\_max
- dist\_weight = 0.5: linear decrease of weight until weight of 0 at d
    = d\_max
- 0.5 \< dist\_weight \< 1: the weighting curve decreases slowly at
    low distances and approaches weight value of 0 at higher distances
    from the fragment, the weight value 0 is reached at d = d\_max
- dist\_weight = 1: no distance weight applied, common static edge
    depth used

### propmap

The *propmap* minimizes the effect of the edge depth and the surrounding
matrix. This has an ecological application if certain landscape features
inside a e.g. forest fragment hamper the human impact (edge effects).  
two methods exist:  

- propmethod=linear: propagated value = actual value - (propmap value
    at this position)  
- propmethod=exponential: propagated value = actual value / (propmap
    value at this position)  

If 0 is chosen using the linear method, then propagated value=actual
value which results in a buffering of the whole region. In order to
minimize the impact the value must be larger than 1. For the exponential
method a value of below 1 should not be chosen, otherwise it will be
propagated infinitely.

## EXAMPLE

An example for the North Carolina sample dataset using class 5 (forest):
For the computation of variable edge effects a costmap is necessary
which need to be defined by the user. Higher costs are resulting in
higher edge depths:

```sh
# class - type - costs
#   1   - developed - 3
#   2   - agriculture - 2
#   3   - herbaceous - 1
#   4   - shrubland - 1
#   5   - forest - 0
#   6   - water - 0
#   7   - sediment - 0

r.mapcalc "costmap_for_corearea = if(landclass96==1,3,if(landclass96==2,2,if(landclass96==3,1,if(landclass96==4,1,if(landclass96==5,0,if(landclass96==6,0,if(landclass96==7,0)))))))"

```

now the edge depth and the resulting core area can be computed:

```sh
r.pi.corearea input=landclass96 costmap=costmap_for_corearea  output=landcover96_corearea keyval=5 buffer=5 distance=5 angle=90 stats=average propmethod=linear
```

the results consist of 2 files:  
landclass96\_corearea: the actual resulting core areas  
landclass96\_corearea\_map: a map showing the edge depths

## SEE ALSO

*[r.pi.grow](r.pi.grow.md), [r.pi.import](r.pi.import.md),
[r.pi.index](r.pi.index.md), [r.pi](r.pi.md)*

## AUTHORS

Programming: Elshad Shirinov  
Scientific concept: Dr. Martin Wegmann  
Department of Remote Sensing  
Remote Sensing and Biodiversity Unit  
University of Wuerzburg, Germany

Port to GRASS GIS 7: Markus Metz

## SOURCE CODE

Available at: [r.pi.corearea source code](https://github.com/OSGeo/grass-addons/tree/grass8/raster/r.pi/r.pi.corearea)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/raster/r.pi/r.pi.corearea))  
Latest change: Friday Feb 21 10:10:05 2025 in commit [7d78fe3](https://github.com/OSGeo/grass-addons/commit/7d78fe34868674c3b6050ba1924e1c5675d155c9)
