---
name: r.pops.spread
description: A dynamic species distribution model for pest or pathogen spread in forest or agricultural ecosystems (PoPS)
keywords: [ raster, spread, model, simulation, disease, pest ]
---

# r.pops.spread

A dynamic species distribution model for pest or pathogen spread in forest or agricultural ecosystems (PoPS)

=== "Command line"

    **r.pops.spread**
    [**-ms**]
    **host**=*name*
    **total_plants**=*name*
    **infected**=*name*
    [**average**=*name*]
    [**average_series**=*basename*]
    [**single_series**=*basename*]
    [**stddev**=*name*]
    [**stddev_series**=*basename*]
    [**probability**=*name*]
    [**probability_series**=*basename*]
    [**outside_spores**=*name*]
    [**spread_rate_output**=*name*]
    **model_type**=*string*
    [**latency_period**=*integer*]
    [**treatments**=*name* [,*name*,...]]
    [**treatment_date**=*string* [,*string*,...]]
    [**treatment_length**=*integer* [,*integer*,...]]
    [**treatment_application**=*string*]
    [**moisture_coefficient_file**=*name*]
    [**temperature_coefficient_file**=*name*]
    [**weather_coefficient_file**=*name*]
    [**lethal_temperature**=*float*]
    [**lethal_month**=*integer*]
    [**temperature_file**=*name*]
    **start_date**=*string*
    **end_date**=*string*
    **seasonality**=*from,to*
    **step_unit**=*string*
    **step_num_units**=*integer*
    [**output_frequency**=*string*]
    [**output_frequency_n**=*integer*]
    [**reproductive_rate**=*float*]
    **natural_dispersal_kernel**=*string*
    **natural_distance**=*float*
    **natural_direction**=*string*
    [**natural_direction_strength**=*float*]
    [**anthropogenic_dispersal_kernel**=*string*]
    [**anthropogenic_distance**=*float*]
    [**anthropogenic_direction**=*string*]
    [**anthropogenic_direction_strength**=*float*]
    [**percent_natural_dispersal**=*float*]
    [**mortality_rate**=*float*]
    [**mortality_time_lag**=*integer*]
    [**mortality_series**=*basename*]
    [**random_seed**=*integer*]
    [**runs**=*integer*]
    [**nprocs**=*integer*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.pops.spread host=name total_plants=name infected=name average=name model_type=SI start_date=string end_date=string seasonality=1,12 step_unit=month step_num_units=1 natural_dispersal_kernel=cauchy natural_distance=0.0 natural_direction=none
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.pops.spread***",
        **host**,
        **total_plants**,
        **infected**,
        **average**=*None*,
        **average_series**=*None*,
        **single_series**=*None*,
        **stddev**=*None*,
        **stddev_series**=*None*,
        **probability**=*None*,
        **probability_series**=*None*,
        **outside_spores**=*None*,
        **spread_rate_output**=*None*,
        **model_type**=*"SI"*,
        **latency_period**=*None*,
        **treatments**=*None*,
        **treatment_date**=*None*,
        **treatment_length**=*None*,
        **treatment_application**=*"ratio_to_all"*,
        **moisture_coefficient_file**=*None*,
        **temperature_coefficient_file**=*None*,
        **weather_coefficient_file**=*None*,
        **lethal_temperature**=*None*,
        **lethal_month**=*None*,
        **temperature_file**=*None*,
        **start_date**,
        **end_date**,
        **seasonality**=*"1,12"*,
        **step_unit**=*"month"*,
        **step_num_units**=*1*,
        **output_frequency**=*"yearly"*,
        **output_frequency_n**=*1*,
        **reproductive_rate**=*4.4*,
        **natural_dispersal_kernel**=*"cauchy"*,
        **natural_distance**,
        **natural_direction**=*"none"*,
        **natural_direction_strength**=*None*,
        **anthropogenic_dispersal_kernel**=*None*,
        **anthropogenic_distance**=*None*,
        **anthropogenic_direction**=*"none"*,
        **anthropogenic_direction_strength**=*None*,
        **percent_natural_dispersal**=*None*,
        **mortality_rate**=*None*,
        **mortality_time_lag**=*None*,
        **mortality_series**=*None*,
        **random_seed**=*None*,
        **runs**=*None*,
        **nprocs**=*None*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.pops.spread", host="name", total_plants="name", infected="name", average="name", model_type="SI", start_date="string", end_date="string", seasonality="1,12", step_unit="month", step_num_units=1, natural_dispersal_kernel="cauchy", natural_distance=0.0, natural_direction="none")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_pops_spread*(**host**,
        **total_plants**,
        **infected**,
        **average**=*None*,
        **average_series**=*None*,
        **single_series**=*None*,
        **stddev**=*None*,
        **stddev_series**=*None*,
        **probability**=*None*,
        **probability_series**=*None*,
        **outside_spores**=*None*,
        **spread_rate_output**=*None*,
        **model_type**=*"SI"*,
        **latency_period**=*None*,
        **treatments**=*None*,
        **treatment_date**=*None*,
        **treatment_length**=*None*,
        **treatment_application**=*"ratio_to_all"*,
        **moisture_coefficient_file**=*None*,
        **temperature_coefficient_file**=*None*,
        **weather_coefficient_file**=*None*,
        **lethal_temperature**=*None*,
        **lethal_month**=*None*,
        **temperature_file**=*None*,
        **start_date**,
        **end_date**,
        **seasonality**=*"1,12"*,
        **step_unit**=*"month"*,
        **step_num_units**=*1*,
        **output_frequency**=*"yearly"*,
        **output_frequency_n**=*1*,
        **reproductive_rate**=*4.4*,
        **natural_dispersal_kernel**=*"cauchy"*,
        **natural_distance**,
        **natural_direction**=*"none"*,
        **natural_direction_strength**=*None*,
        **anthropogenic_dispersal_kernel**=*None*,
        **anthropogenic_distance**=*None*,
        **anthropogenic_direction**=*"none"*,
        **anthropogenic_direction_strength**=*None*,
        **percent_natural_dispersal**=*None*,
        **mortality_rate**=*None*,
        **mortality_time_lag**=*None*,
        **mortality_series**=*None*,
        **random_seed**=*None*,
        **runs**=*None*,
        **nprocs**=*None*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_pops_spread(host="name", total_plants="name", infected="name", average="name", model_type="SI", start_date="string", end_date="string", seasonality="1,12", step_unit="month", step_num_units=1, natural_dispersal_kernel="cauchy", natural_distance=0.0, natural_direction="none")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **host**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Input host raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of hosts per cell.  
    **total_plants**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Input raster map of total plants  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of all plants per cell  
    **infected**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Input raster map of initial infection  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of infected hosts per cell  
    **average**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Average infected across multiple runs  
    **average_series**=*basename*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename for output series of average infected across multiple runs  
    **single_series**=*basename*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename for output series of infected as single stochastic run  
    **stddev**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Standard deviations  
    **stddev_series**=*basename*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename for output series of standard deviations  
    **probability**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Infection probability (in percent)  
    **probability_series**=*basename*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename for output series of probabilities  
    **outside_spores**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output vector map of spores or pest units outside of modeled area  
    **spread_rate_output**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output CSV file containg yearly spread rate in N, S, E, W directions  
    **model_type**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Epidemiological model type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *SI, SEI*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *SI*  
    &nbsp;&nbsp;&nbsp;&nbsp;**SI**: Susceptible-infected epidemiological model  
    &nbsp;&nbsp;&nbsp;&nbsp;**SEI**: Susceptible-exposed-infected epidemiological model (uses latency_period)  
    **latency_period**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Latency period in simulation steps  
    &nbsp;&nbsp;&nbsp;&nbsp;How long it takes for a hosts to become infected after being exposed (unit is a simulation step)  
    **treatments**=*name* [,*name*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map(s) of treatments (treated 1, otherwise 0)  
    **treatment_date**=*string* [,*string*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Dates when treatments are applied (e.g. 2020-01-15)  
    **treatment_length**=*integer* [,*integer*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Treatment length in days  
    &nbsp;&nbsp;&nbsp;&nbsp;Treatment length 0 results in simple removal of host, length &gt; 0 makes host resistant for certain number of days  
    **treatment_application**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Type of treatmet application  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *ratio_to_all, all_infected_in_cell*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *ratio_to_all*  
    **moisture_coefficient_file**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input file with one moisture coefficient map name per line  
    &nbsp;&nbsp;&nbsp;&nbsp;Moisture coefficient  
    **temperature_coefficient_file**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input file with one temperature coefficient map name per line  
    &nbsp;&nbsp;&nbsp;&nbsp;Temperature coefficient  
    **weather_coefficient_file**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input file with one weather coefficient map name per line  
    &nbsp;&nbsp;&nbsp;&nbsp;Weather coefficient  
    **lethal_temperature**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Temperature at which the pest or pathogen dies  
    &nbsp;&nbsp;&nbsp;&nbsp;The temerature unit must be the same as for thetemerature raster map (typically degrees of Celsius)  
    **lethal_month**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Month when the pest or patogen dies due to low temperature  
    &nbsp;&nbsp;&nbsp;&nbsp;The temperature unit must be the same as for thetemperature raster map (typically degrees of Celsius)  
    **temperature_file**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input file with one temperature raster map name per line  
    &nbsp;&nbsp;&nbsp;&nbsp;The temperature should be in actual temperature units (typically degrees of Celsius)  
    **start_date**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Start date of the simulation in YYYY-MM-DD format  
    **end_date**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;End date of the simulation in YYYY-MM-DD format  
    **seasonality**=*from,to* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Seasonal spread (from,to)  
    &nbsp;&nbsp;&nbsp;&nbsp;Spread limited to certain months (season), for example 5,9 for spread starting at the beginning of May and ending at the end of September  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1,12*  
    **step_unit**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Unit of simulation steps  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *day, week, month*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *month*  
    &nbsp;&nbsp;&nbsp;&nbsp;**day**: Compute next simulation step every N days  
    &nbsp;&nbsp;&nbsp;&nbsp;**week**: Compute next simulation step every N weeks  
    &nbsp;&nbsp;&nbsp;&nbsp;**month**: Compute next simulation step every N months  
    **step_num_units**=*integer* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of days/weeks/months in each step  
    &nbsp;&nbsp;&nbsp;&nbsp;Step is given by number and unit, e.g. step_num_units=5 and step_unit=day means step is 5 days  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-100*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **output_frequency**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Frequency of simulation output  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *yearly, monthly, weekly, daily, every_n_steps*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *yearly*  
    **output_frequency_n**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output frequency every N steps  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-100*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **reproductive_rate**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of spores or pest units produced by a single host  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of spores or pest units produced by a single host under optimal weather conditions  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *4.4*  
    **natural_dispersal_kernel**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Natural dispersal kernel type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *cauchy, exponential*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *cauchy*  
    **natural_distance**=*float* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Distance parameter for natural dispersal kernel  
    **natural_direction**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Direction of natural dispersal kernel  
    &nbsp;&nbsp;&nbsp;&nbsp;Typically prevailing wind direction; none means that there is no directionality or no wind  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *N, NE, E, SE, S, SW, W, NW, NONE, none*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *none*  
    **natural_direction_strength**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Strength of direction of natural dispersal kernel  
    &nbsp;&nbsp;&nbsp;&nbsp;The kappa parameter of von Mises distribution (concentration); typically the strength of the wind direction  
    **anthropogenic_dispersal_kernel**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Anthropogenic dispersal kernel type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *cauchy, exponential*  
    **anthropogenic_distance**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Distance parameter for anthropogenic dispersal kernel  
    **anthropogenic_direction**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Direction of anthropogenic dispersal kernel  
    &nbsp;&nbsp;&nbsp;&nbsp;Value none means that there is no directionality  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *N, NE, E, SE, S, SW, W, NW, NONE, none*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *none*  
    **anthropogenic_direction_strength**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Strength of direction of anthropogenic dispersal kernel  
    &nbsp;&nbsp;&nbsp;&nbsp;The kappa parameter of von Mises distribution (concentration); typically the strength of the wind direction  
    **percent_natural_dispersal**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Percentage of natural dispersal  
    &nbsp;&nbsp;&nbsp;&nbsp;How often is the natural dispersal kernel used versus the anthropogenic dispersal kernel  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0-1*  
    **mortality_rate**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Mortality rate of infected hosts  
    &nbsp;&nbsp;&nbsp;&nbsp;Percentage of infected hosts that die in a given year (hosts are removed from the infected pool)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0-1*  
    **mortality_time_lag**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Time lag from infection until mortality can occur in years  
    &nbsp;&nbsp;&nbsp;&nbsp;How many years it takes for an infected host to die (value 1 for hosts dying at the end of the first year)  
    **mortality_series**=*basename*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename for series of number of dead hosts  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename for output series of number of dead hosts (requires mortality to be activated)  
    **random_seed**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Seed for random number generator  
    &nbsp;&nbsp;&nbsp;&nbsp;The same seed can be used to obtain same results or random seed can be generated by other means.  
    **runs**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of simulation runs  
    &nbsp;&nbsp;&nbsp;&nbsp;The individual runs will obtain different seeds and will be averaged for the output  
    **nprocs**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of threads for parallel computing  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-*  
    **-m**  
    &nbsp;&nbsp;&nbsp;&nbsp;Apply mortality  
    &nbsp;&nbsp;&nbsp;&nbsp;After certain number of years, start removing dead hosts from the infected pool with a given rate  
    **-s**  
    &nbsp;&nbsp;&nbsp;&nbsp;Generate random seed (result is non-deterministic)  
    &nbsp;&nbsp;&nbsp;&nbsp;Automatically generates random seed for random number generator (use when you don't want to provide the seed option)  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **host** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input host raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of hosts per cell.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **total_plants** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input raster map of total plants  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of all plants per cell  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **infected** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input raster map of initial infection  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of infected hosts per cell  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **average** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Average infected across multiple runs  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **average_series** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename for output series of average infected across multiple runs  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *basename*  
    **single_series** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename for output series of infected as single stochastic run  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *basename*  
    **stddev** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Standard deviations  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **stddev_series** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename for output series of standard deviations  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *basename*  
    **probability** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Infection probability (in percent)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **probability_series** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename for output series of probabilities  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *basename*  
    **outside_spores** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output vector map of spores or pest units outside of modeled area  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, vector, *name*  
    **spread_rate_output** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output CSV file containg yearly spread rate in N, S, E, W directions  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    **model_type** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Epidemiological model type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *SI, SEI*  
    &nbsp;&nbsp;&nbsp;&nbsp;**SI**: Susceptible-infected epidemiological model  
    &nbsp;&nbsp;&nbsp;&nbsp;**SEI**: Susceptible-exposed-infected epidemiological model (uses latency_period)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *SI*  
    **latency_period** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Latency period in simulation steps  
    &nbsp;&nbsp;&nbsp;&nbsp;How long it takes for a hosts to become infected after being exposed (unit is a simulation step)  
    **treatments** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map(s) of treatments (treated 1, otherwise 0)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **treatment_date** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Dates when treatments are applied (e.g. 2020-01-15)  
    **treatment_length** : int | list[int] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Treatment length in days  
    &nbsp;&nbsp;&nbsp;&nbsp;Treatment length 0 results in simple removal of host, length &gt; 0 makes host resistant for certain number of days  
    **treatment_application** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Type of treatmet application  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *ratio_to_all, all_infected_in_cell*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *ratio_to_all*  
    **moisture_coefficient_file** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input file with one moisture coefficient map name per line  
    &nbsp;&nbsp;&nbsp;&nbsp;Moisture coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **temperature_coefficient_file** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input file with one temperature coefficient map name per line  
    &nbsp;&nbsp;&nbsp;&nbsp;Temperature coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **weather_coefficient_file** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input file with one weather coefficient map name per line  
    &nbsp;&nbsp;&nbsp;&nbsp;Weather coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **lethal_temperature** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Temperature at which the pest or pathogen dies  
    &nbsp;&nbsp;&nbsp;&nbsp;The temerature unit must be the same as for thetemerature raster map (typically degrees of Celsius)  
    **lethal_month** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Month when the pest or patogen dies due to low temperature  
    &nbsp;&nbsp;&nbsp;&nbsp;The temperature unit must be the same as for thetemperature raster map (typically degrees of Celsius)  
    **temperature_file** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input file with one temperature raster map name per line  
    &nbsp;&nbsp;&nbsp;&nbsp;The temperature should be in actual temperature units (typically degrees of Celsius)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **start_date** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Start date of the simulation in YYYY-MM-DD format  
    **end_date** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;End date of the simulation in YYYY-MM-DD format  
    **seasonality** : tuple[str, str] | list[str] | str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Seasonal spread (from,to)  
    &nbsp;&nbsp;&nbsp;&nbsp;Spread limited to certain months (season), for example 5,9 for spread starting at the beginning of May and ending at the end of September  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *from,to*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1,12*  
    **step_unit** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Unit of simulation steps  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *day, week, month*  
    &nbsp;&nbsp;&nbsp;&nbsp;**day**: Compute next simulation step every N days  
    &nbsp;&nbsp;&nbsp;&nbsp;**week**: Compute next simulation step every N weeks  
    &nbsp;&nbsp;&nbsp;&nbsp;**month**: Compute next simulation step every N months  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *month*  
    **step_num_units** : int, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of days/weeks/months in each step  
    &nbsp;&nbsp;&nbsp;&nbsp;Step is given by number and unit, e.g. step_num_units=5 and step_unit=day means step is 5 days  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-100*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **output_frequency** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Frequency of simulation output  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *yearly, monthly, weekly, daily, every_n_steps*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *yearly*  
    **output_frequency_n** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output frequency every N steps  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-100*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **reproductive_rate** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of spores or pest units produced by a single host  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of spores or pest units produced by a single host under optimal weather conditions  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *4.4*  
    **natural_dispersal_kernel** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Natural dispersal kernel type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *cauchy, exponential*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *cauchy*  
    **natural_distance** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Distance parameter for natural dispersal kernel  
    **natural_direction** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Direction of natural dispersal kernel  
    &nbsp;&nbsp;&nbsp;&nbsp;Typically prevailing wind direction; none means that there is no directionality or no wind  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *N, NE, E, SE, S, SW, W, NW, NONE, none*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *none*  
    **natural_direction_strength** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Strength of direction of natural dispersal kernel  
    &nbsp;&nbsp;&nbsp;&nbsp;The kappa parameter of von Mises distribution (concentration); typically the strength of the wind direction  
    **anthropogenic_dispersal_kernel** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Anthropogenic dispersal kernel type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *cauchy, exponential*  
    **anthropogenic_distance** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Distance parameter for anthropogenic dispersal kernel  
    **anthropogenic_direction** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Direction of anthropogenic dispersal kernel  
    &nbsp;&nbsp;&nbsp;&nbsp;Value none means that there is no directionality  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *N, NE, E, SE, S, SW, W, NW, NONE, none*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *none*  
    **anthropogenic_direction_strength** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Strength of direction of anthropogenic dispersal kernel  
    &nbsp;&nbsp;&nbsp;&nbsp;The kappa parameter of von Mises distribution (concentration); typically the strength of the wind direction  
    **percent_natural_dispersal** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Percentage of natural dispersal  
    &nbsp;&nbsp;&nbsp;&nbsp;How often is the natural dispersal kernel used versus the anthropogenic dispersal kernel  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0-1*  
    **mortality_rate** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Mortality rate of infected hosts  
    &nbsp;&nbsp;&nbsp;&nbsp;Percentage of infected hosts that die in a given year (hosts are removed from the infected pool)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0-1*  
    **mortality_time_lag** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Time lag from infection until mortality can occur in years  
    &nbsp;&nbsp;&nbsp;&nbsp;How many years it takes for an infected host to die (value 1 for hosts dying at the end of the first year)  
    **mortality_series** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename for series of number of dead hosts  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename for output series of number of dead hosts (requires mortality to be activated)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *basename*  
    **random_seed** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Seed for random number generator  
    &nbsp;&nbsp;&nbsp;&nbsp;The same seed can be used to obtain same results or random seed can be generated by other means.  
    **runs** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of simulation runs  
    &nbsp;&nbsp;&nbsp;&nbsp;The individual runs will obtain different seeds and will be averaged for the output  
    **nprocs** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of threads for parallel computing  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *m*, *s*  
    &nbsp;&nbsp;&nbsp;&nbsp;**m**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Apply mortality  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;After certain number of years, start removing dead hosts from the infected pool with a given rate  
    &nbsp;&nbsp;&nbsp;&nbsp;**s**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Generate random seed (result is non-deterministic)  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Automatically generates random seed for random number generator (use when you don't want to provide the seed option)  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **host** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input host raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of hosts per cell.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **total_plants** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input raster map of total plants  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of all plants per cell  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **infected** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input raster map of initial infection  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of infected hosts per cell  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **average** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Average infected across multiple runs  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **average_series** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename for output series of average infected across multiple runs  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *basename*  
    **single_series** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename for output series of infected as single stochastic run  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *basename*  
    **stddev** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Standard deviations  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **stddev_series** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename for output series of standard deviations  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *basename*  
    **probability** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Infection probability (in percent)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **probability_series** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename for output series of probabilities  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *basename*  
    **outside_spores** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output vector map of spores or pest units outside of modeled area  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, vector, *name*  
    **spread_rate_output** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output CSV file containg yearly spread rate in N, S, E, W directions  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    **model_type** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Epidemiological model type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *SI, SEI*  
    &nbsp;&nbsp;&nbsp;&nbsp;**SI**: Susceptible-infected epidemiological model  
    &nbsp;&nbsp;&nbsp;&nbsp;**SEI**: Susceptible-exposed-infected epidemiological model (uses latency_period)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *SI*  
    **latency_period** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Latency period in simulation steps  
    &nbsp;&nbsp;&nbsp;&nbsp;How long it takes for a hosts to become infected after being exposed (unit is a simulation step)  
    **treatments** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map(s) of treatments (treated 1, otherwise 0)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **treatment_date** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Dates when treatments are applied (e.g. 2020-01-15)  
    **treatment_length** : int | list[int] | str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Treatment length in days  
    &nbsp;&nbsp;&nbsp;&nbsp;Treatment length 0 results in simple removal of host, length &gt; 0 makes host resistant for certain number of days  
    **treatment_application** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Type of treatmet application  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *ratio_to_all, all_infected_in_cell*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *ratio_to_all*  
    **moisture_coefficient_file** : str | io.StringIO, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input file with one moisture coefficient map name per line  
    &nbsp;&nbsp;&nbsp;&nbsp;Moisture coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **temperature_coefficient_file** : str | io.StringIO, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input file with one temperature coefficient map name per line  
    &nbsp;&nbsp;&nbsp;&nbsp;Temperature coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **weather_coefficient_file** : str | io.StringIO, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input file with one weather coefficient map name per line  
    &nbsp;&nbsp;&nbsp;&nbsp;Weather coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **lethal_temperature** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Temperature at which the pest or pathogen dies  
    &nbsp;&nbsp;&nbsp;&nbsp;The temerature unit must be the same as for thetemerature raster map (typically degrees of Celsius)  
    **lethal_month** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Month when the pest or patogen dies due to low temperature  
    &nbsp;&nbsp;&nbsp;&nbsp;The temperature unit must be the same as for thetemperature raster map (typically degrees of Celsius)  
    **temperature_file** : str | io.StringIO, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input file with one temperature raster map name per line  
    &nbsp;&nbsp;&nbsp;&nbsp;The temperature should be in actual temperature units (typically degrees of Celsius)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **start_date** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Start date of the simulation in YYYY-MM-DD format  
    **end_date** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;End date of the simulation in YYYY-MM-DD format  
    **seasonality** : tuple[str, str] | list[str] | str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Seasonal spread (from,to)  
    &nbsp;&nbsp;&nbsp;&nbsp;Spread limited to certain months (season), for example 5,9 for spread starting at the beginning of May and ending at the end of September  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *from,to*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1,12*  
    **step_unit** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Unit of simulation steps  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *day, week, month*  
    &nbsp;&nbsp;&nbsp;&nbsp;**day**: Compute next simulation step every N days  
    &nbsp;&nbsp;&nbsp;&nbsp;**week**: Compute next simulation step every N weeks  
    &nbsp;&nbsp;&nbsp;&nbsp;**month**: Compute next simulation step every N months  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *month*  
    **step_num_units** : int, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of days/weeks/months in each step  
    &nbsp;&nbsp;&nbsp;&nbsp;Step is given by number and unit, e.g. step_num_units=5 and step_unit=day means step is 5 days  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-100*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **output_frequency** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Frequency of simulation output  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *yearly, monthly, weekly, daily, every_n_steps*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *yearly*  
    **output_frequency_n** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output frequency every N steps  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-100*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **reproductive_rate** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of spores or pest units produced by a single host  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of spores or pest units produced by a single host under optimal weather conditions  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *4.4*  
    **natural_dispersal_kernel** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Natural dispersal kernel type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *cauchy, exponential*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *cauchy*  
    **natural_distance** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Distance parameter for natural dispersal kernel  
    **natural_direction** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Direction of natural dispersal kernel  
    &nbsp;&nbsp;&nbsp;&nbsp;Typically prevailing wind direction; none means that there is no directionality or no wind  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *N, NE, E, SE, S, SW, W, NW, NONE, none*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *none*  
    **natural_direction_strength** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Strength of direction of natural dispersal kernel  
    &nbsp;&nbsp;&nbsp;&nbsp;The kappa parameter of von Mises distribution (concentration); typically the strength of the wind direction  
    **anthropogenic_dispersal_kernel** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Anthropogenic dispersal kernel type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *cauchy, exponential*  
    **anthropogenic_distance** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Distance parameter for anthropogenic dispersal kernel  
    **anthropogenic_direction** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Direction of anthropogenic dispersal kernel  
    &nbsp;&nbsp;&nbsp;&nbsp;Value none means that there is no directionality  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *N, NE, E, SE, S, SW, W, NW, NONE, none*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *none*  
    **anthropogenic_direction_strength** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Strength of direction of anthropogenic dispersal kernel  
    &nbsp;&nbsp;&nbsp;&nbsp;The kappa parameter of von Mises distribution (concentration); typically the strength of the wind direction  
    **percent_natural_dispersal** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Percentage of natural dispersal  
    &nbsp;&nbsp;&nbsp;&nbsp;How often is the natural dispersal kernel used versus the anthropogenic dispersal kernel  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0-1*  
    **mortality_rate** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Mortality rate of infected hosts  
    &nbsp;&nbsp;&nbsp;&nbsp;Percentage of infected hosts that die in a given year (hosts are removed from the infected pool)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *0-1*  
    **mortality_time_lag** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Time lag from infection until mortality can occur in years  
    &nbsp;&nbsp;&nbsp;&nbsp;How many years it takes for an infected host to die (value 1 for hosts dying at the end of the first year)  
    **mortality_series** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename for series of number of dead hosts  
    &nbsp;&nbsp;&nbsp;&nbsp;Basename for output series of number of dead hosts (requires mortality to be activated)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *basename*  
    **random_seed** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Seed for random number generator  
    &nbsp;&nbsp;&nbsp;&nbsp;The same seed can be used to obtain same results or random seed can be generated by other means.  
    **runs** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of simulation runs  
    &nbsp;&nbsp;&nbsp;&nbsp;The individual runs will obtain different seeds and will be averaged for the output  
    **nprocs** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of threads for parallel computing  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *m*, *s*  
    &nbsp;&nbsp;&nbsp;&nbsp;**m**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Apply mortality  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;After certain number of years, start removing dead hosts from the infected pool with a given rate  
    &nbsp;&nbsp;&nbsp;&nbsp;**s**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Generate random seed (result is non-deterministic)  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Automatically generates random seed for random number generator (use when you don't want to provide the seed option)  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

Module *r.pops.spread* is a dynamic species distribution model for pest
or pathogen spread in forest or agricultural ecosystems. The model is
process based meaning that it uses understanding of the effect of
weather on reproduction and survival of the pest or pathogen in order to
simulate spread of the pest or pathogen into the future using
simulation.

[![image-alt](r_pops_spread.png)](r_pops_spread.png)  
*Figure: Infected hosts in a landscape, a typical model result*

### About the model

The module is using the PoPS Core library which is implementing the PoPS
model and it is a central part of the [Pest or Pathogen Spread
(PoPS)](https://popsmodel.org/) project.

[![image-alt](pops_logo.png)](pops_logo.png)  
*Figure: Logo of the Pest or Pathogen Spread framework the PoPS is part
of*

The PoPS model is a stochastic spread model of pests and pathogens in
forest and agricultural landscapes. It is used for various pest,
pathogens, and hosts including animals, not just plants, as hosts. It
was originally developed for *Phytophthora ramorum* and the original
version of the model was written in R, later with Rcpp (Tonini, 2017),
and was based on Meentemeyer (2011) paper.

The current implementation of the GRASS GIS module is using PoPS Core
header-only C++ library which implements the PoPS model. The primary
development of [PoPS
Core](https://github.com/ncsu-landscape-dynamics/pops-core) and of [this
module](https://github.com/ncsu-landscape-dynamics/r.pops.spread)
happens in a separate repositories and GRASS GIS Addons repository
contains the latest release of the model. An alternative [steering
version](https://github.com/ncsu-landscape-dynamics/r.pops.spread/tree/steering)
of this module exists which includes a set of features supporting
geospatial simulation steering (Petrasova, 2020) which is useful for
exploring adaptive management scenarios.

### Model parameters

Two basic epidemiological model types (**model\_type**) are available
for a transition of hosts between susceptible and infected classes: 1)
susceptible-infected (SI) for an immediate transition when a disperser
establishes on the host and 2) susceptible-exposed-infected (SEI) for an
intermediate state when the host first becomes exposed and only after a
latency period (**latency\_period**) is over. This page lists above the
numerous inputs and parameters, although many of them have default
values, some need careful consideration and calibration. The best way
how to identify options relevant to a given use case is to go through
one of the available tutorials.

### Calibration

Typically, the model needs to be calibrated. You can obtain the
calibration from a published work, colleague, calibrate the model
manually (in GRASS GIS), or use the R interface to PoPS called
[rpops](https://github.com/ncsu-landscape-dynamics/rpops) which has
dedicated functions for calibration.

## NOTES

- The directions of wind consider north (N) to be grid north, if your
    true north is different direction, you need to make an adjustment.
- The module currently does not handle NULL (no data) as input, so you
    need to change the NULLs to (most likely) zeros, for example:
    `r.null map=infection null=0`.

## EXAMPLES

### Obtaining list of rasters

Use [R
script](https://github.com/ncsu-landscape-dynamics/weather-coefficient)
to create weather coefficients based on a defined polynomial.

Example of creating file with list of input maps (unix-like command
line):

```sh
g.list type=raster pattern="moisture_*" mapset=climate -m > moistures.txt
g.list type=raster pattern="temperature_*" mapset=climate -m > temperatures.txt
```

Note that the above assumes that the names will be ordered by time. This
will happen automatically if they are, e.g. numbered as 001, 002, etc.
(e.g. `temperature_001` and not `temperature_1`). If they are numbered
without the zero-padding, i.e. 1, 2, ..., 10, 11, ..., then in a
unix-like command line, you can do pipe the result through *sort* with
`-n` (`| sort -n`). For example, for map names like `temperature_1`, the
following unix-like command will do the sorting:

```sh
g.list type=raster pattern="temperature_*" mapset=climate | sort -k2 -t_ -n > temperatures.txt
```

Note the underscore which tells sort where to split the name for sorting
and the number 2 which indicates which part of the name to use for
sorting after splitting. If you have the weather-related timeseries in a
separate mapset, you can add this mapset to the search path of your
current mapset so that you can have the rasters in the list without
specifying the mapset. To add to the search path, use for example:

```sh
g.mapsets mapset=climate
```

### Generating a constant coefficient

In case the moisture coefficient is not used, we can generate a constant
raster map to be used as the coefficient:

```sh
r.mapcalc "const_1 = 1"
```

Then using unix-like command line, we can create a list of these rasters
in a file based on the number of lines in a temperature list files we
created earlier:

```sh
NUM_LINES=`cat temperatures.txt | wc -l`
echo const_1 > moistures.txt
for LINE in `seq 2 $NUM_LINES`; do echo const_1 >> moistures.txt; done;
```

### Creating treatments

To account for (vector) treatments partially covering host cells:

```sh
# set resolution for treatments and convert to raster
g.region res=10 -ap
v.to.rast input=treatment output=treatment use=val

# resample to lower resolution (match host map resolution)
g.region align=host_map -p
r.resamp.stats -w input=treatment output=treatment_resampled method=count
# get maximum value, which is dependent on resolution
# e.g. when resampling from 10m to 100m, max will be 100 (100 small cells in 1 big cell)
r.info -r treatment_resampled
# result will be 0 to 1
r.mapcalc "treatment_float = test_treatment_resampled / 100"
# adjust host layer
r.mapcalc "treated_host = host - host * treatment_float"
```

### Running the model

Example of the run of the model (unix-like command line):

```sh
r.pops.spread host=host total_plants=all infected=infected_2005 \
    moisture_coefficient_file=moistures.txt temperature_coefficient_file=temperatures.txt \
    output=spread step=week start_time=2005 end_time=2010 \
    reproductive_rate=4 dispersal_kernel=cauchy wind=NE random_seed=4
```

## REFERENCES

To cite this module, please refer to [How to
cite](https://github.com/ncsu-landscape-dynamics/r.pops.spread#how-to-cite)
section in the readme file.

- Meentemeyer, Ross K., Nik J. Cunniffe, Alex R. Cook, Joao A. N.
    Filipe, Richard D. Hunter, David M. Rizzo, and Christopher A.
    Gilligan, 2011. Epidemiological modeling of invasion in
    heterogeneous landscapes: spread of sudden oak death in California
    (1990-2030). *Ecosphere* 2:art17.
    [DOI: 10.1890/ES10-00192.1](https://doi.org/10.1890/ES10-00192.1)
- Tonini, Francesco, Douglas Shoemaker, Anna Petrasova, Brendan
    Harmon, Vaclav Petras, Richard C. Cobb, Helena Mitasova, and Ross K.
    Meentemeyer. Tangible geospatial modeling for collaborative
    solutions to invasive species management. *Environmental Modelling &
    Software* 92 (2017): 176-188.
    [DOI: 10.1016/j.envsoft.2017.02.020](https://doi.org/10.1016/j.envsoft.2017.02.020)
- Petrasova, A., Gaydos, D.A., Petras, V., Jones, C.M., Mitasova, H.
    and Meentemeyer, R.K., 2020. Geospatial simulation steering for
    adaptive management. *Environmental Modelling & Software* 133:
    104801.
    [DOI: 10.1016/j.envsoft.2020.104801](https://doi.org/10.1016/j.envsoft.2020.104801)

## SEE ALSO

*[r.spread](https://grass.osgeo.org/grass-stable/manuals/r.spread.html),
[r.grow](https://grass.osgeo.org/grass-stable/manuals/r.grow.html),
[r.lake](https://grass.osgeo.org/grass-stable/manuals/r.lake.html),
[r.futures](r.futures.md)*

Tutorials and other resources:

- [PoPS Model Introduction in GRASS Using Jupyter
    Notebook](https://github.com/ncsu-landscape-dynamics/pops-intro-grass-notebook/blob/master/pops-intro-grass-notebook.ipynb)
    [![image-alt](https://mybinder.org/badge_logo.svg)](https://mybinder.org/v2/gh/ncsu-landscape-dynamics/pops-intro-grass-notebook/master?urlpath=lab/tree/pops-intro-grass-notebook.ipynb "Go to an interactive notebook in Binder")
- [SOD Spread
    tutorial](https://grasswiki.osgeo.org/wiki/SOD_Spread_tutorial) (at
    GRASS wiki)

## AUTHORS

In alphabetical order:

Chris Jones\*  
Margaret Lawrimore\*  
Vaclav Petras\*  
Anna Petrasova\*  

Previous contributors:

Zexi Chen\*  
Devon Gaydos\*  
Francesco Tonini\*  
  
\* [Center for Geospatial Analytics, NC State
University](https://cnr.ncsu.edu/geospatial)

## SOURCE CODE

Available at: [r.pops.spread source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/raster/r.pops.spread)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/raster/r.pops.spread))  
Latest change: Sunday Aug 03 16:38:39 2025 in commit [bb03c10](https://github.com/OSGeo/grass-addons/commit/bb03c1004c651365424c3f795b9519e02aabfe42)
