---
name: r.forestfrag.py
description: Computes the forest fragmentation index (Riitters et al. 2000)
keywords: [ raster, landscape structure analysis, forest, fragmentation index, Riitters ]
---

# r.forestfrag.py

Computes the forest fragmentation index (Riitters et al. 2000)

=== "Command line"

    **r.forestfrag.py**
    [**-rtsa**]
    **input**=*name*
    **output**=*name*
    [**size**=*number*]
    [**pf**=*name*]
    [**pff**=*name*]
    [**window**=*integer*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.forestfrag.py input=name output=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.forestfrag.py***",
        **input**,
        **output**,
        **size**=*3*,
        **pf**=*None*,
        **pff**=*None*,
        **window**=*None*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.forestfrag.py", input="name", output="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_forestfrag_py*(**input**,
        **output**,
        **size**=*3*,
        **pf**=*None*,
        **pff**=*None*,
        **window**=*None*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_forestfrag_py(input="name", output="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of forest raster map (where forest=1, non-forest=0)  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    **size**=*number*  
    &nbsp;&nbsp;&nbsp;&nbsp;Moving window size (odd number)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *3-*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *3*  
    **pf**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output Pf (forest area density) raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Proportion of area which is forested (amount of forest)  
    **pff**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output Pff (forest connectivity) raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Conditional probability that neighboring cell is forest  
    **window**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;This option is deprecated, use the option size instead  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *3-*  
    **-r**  
    &nbsp;&nbsp;&nbsp;&nbsp;Set computational region to input raster map  
    **-t**  
    &nbsp;&nbsp;&nbsp;&nbsp;Keep Pf and Pff maps  
    **-s**  
    &nbsp;&nbsp;&nbsp;&nbsp;Run r.report on output map  
    **-a**  
    &nbsp;&nbsp;&nbsp;&nbsp;Trim the output map to avoid border effects  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of forest raster map (where forest=1, non-forest=0)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **size** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Moving window size (odd number)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *number*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *3-*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *3*  
    **pf** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output Pf (forest area density) raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Proportion of area which is forested (amount of forest)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **pff** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output Pff (forest connectivity) raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Conditional probability that neighboring cell is forest  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **window** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;This option is deprecated, use the option size instead  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *3-*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *r*, *t*, *s*, *a*  
    &nbsp;&nbsp;&nbsp;&nbsp;**r**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Set computational region to input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;**t**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Keep Pf and Pff maps  
    &nbsp;&nbsp;&nbsp;&nbsp;**s**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Run r.report on output map  
    &nbsp;&nbsp;&nbsp;&nbsp;**a**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Trim the output map to avoid border effects  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of forest raster map (where forest=1, non-forest=0)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **size** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Moving window size (odd number)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *number*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *3-*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *3*  
    **pf** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output Pf (forest area density) raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Proportion of area which is forested (amount of forest)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **pff** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output Pff (forest connectivity) raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Conditional probability that neighboring cell is forest  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **window** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;This option is deprecated, use the option size instead  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *3-*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *r*, *t*, *s*, *a*  
    &nbsp;&nbsp;&nbsp;&nbsp;**r**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Set computational region to input raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;**t**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Keep Pf and Pff maps  
    &nbsp;&nbsp;&nbsp;&nbsp;**s**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Run r.report on output map  
    &nbsp;&nbsp;&nbsp;&nbsp;**a**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Trim the output map to avoid border effects  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.forestfrag* Computes the forest fragmentation following the
methodology proposed by Riitters et al. (2000). See [this
article](https://www.ecologyandsociety.org/vol4/iss2/art3/) for a
detailed explanation.

It follows a "sliding window" algorithm with overlapping windows. The
amount of forest and its occurrence as adjacent forest pixels within
fixed-area "moving-windows" surrounding each forest pixel is measured.
The window size is user-defined. The result is stored at the location of
the center pixel. Thus, a pixel value in the derived map refers to
"between-pixel" fragmentation around the corresponding forest location.

As input it requires a binary map with (1) forest and (0) non-forest.
Obviously, one can replace forest any other land cover type. If one
wants to exclude the influence of a specific land cover type, e.g.,
water bodies, it should be classified as no-data (NA) in the input map.
See e.g., [blog
post](https://pvanb.wordpress.com/2016/03/25/update-of-r-forestfrag-addon/).

Let *Pf* be the proportion of pixels in the window that are forested.
Define *Pff* (strictly) as the proportion of all adjacent (cardinal
directions only) pixel pairs that include at least one forest pixel, for
which both pixels are forested. *Pff* thus (roughly) estimates the
conditional probability that, given a pixel of forest, its neighbor is
also forest. The classification model then identifies six fragmentation
categories as:

```text
interior:       Pf = 1.0
patch:          Pf < 0.4
transitional:   0.4 ≤ Pf < 0.6
edge:           Pf ≥ 0.6 and Pf - Pff < 0
perforated:     Pf ≥ 0.6 and Pf - Pff > 0
undetermined:   Pf ≥ 0.6 and Pf = Pff
```

## NOTES

- The moving window size is user-defined (default=3) and must be an
    odd number. If an even number is given, the function will stop with
    an error message.
- No-data cells are ignored. This means that statistics at the raster
    edges are based on fewer cells (smaller) moving windows. If this is
    a problem, the user can choose to have the output raster trimmed
    with a number of raster cells equal to 1/2 \* the size of the moving
    window.
- The function respects the region. However, the user has the option
    to set the region to match the input layer.

## EXAMPLE

In the North Carolina sample Location, set the computational region to
match the land classification raster map:

```sh
g.region raster=landclass96
```

Then mark all cells which are forest as 1 and everything else as zero:

```sh
r.mapcalc "forest = if(landclass96 == 5, 1, 0)"
```

Use the new forest presence raster map to compute the forest
fragmentation index with window size 7:

```sh
r.forestfrag input=forest output=fragmentation window=7
```

![image-alt](r_forestfrag_window_7.png)
![image-alt](r_forestfrag_window_11.png)

*Two forest fragmentation indices with window size 7 (left) and 11
(right) show how increasing window size increases the amount of edges.*

## SEE ALSO

*[r.mapcalc](https://grass.osgeo.org/grass-stable/manuals/r.mapcalc.html),
[r.li](https://grass.osgeo.org/grass-stable/manuals/r.li.html)*

The addon is based on the
[r.forestfrag.sh](https://grasswiki.osgeo.org/wiki/AddOns/GRASS_6#r.forestfrag)
script, with as extra options user-defined moving window size, option to
trim the region (by default it respects the region) and a better
handling of no-data cells.

## REFERENCES

Riitters, K., J. Wickham, R. O'Neill, B. Jones, and E. Smith. 2000.
Global-scale patterns of forest fragmentation. Conservation Ecology
4(2): 3. \[online\] URL: <https://www.consecol.org/vol4/iss2/art3/>

## AUTHORS

Emmanuel Sambale (original shell version)  
Stefan Sylla (original shell version)  
Paulo van Breugel (Python version, user-defined moving window size)  
Vaclav Petras (major code clean up)

## SOURCE CODE

Available at: [r.forestfrag source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/raster/r.forestfrag)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/raster/r.forestfrag))  
Latest change: Thursday Mar 20 21:36:57 2025 in commit [7286ecf](https://github.com/OSGeo/grass-addons/commit/7286ecf7af235bfd089fb9b1b82fb383cf95f3fc)
