---
name: r.hydro.flatten.py
description: Derive elevation of water bodies for hydro-flattening
keywords: [ raster, elevation, hydrology, lidar, LIDAR ]
---

# r.hydro.flatten.py

Derive elevation of water bodies for hydro-flattening

=== "Command line"

    **r.hydro.flatten.py**
    [**-k**]
    **input**=*name*
    [**breaklines**=*name*]
    **water_elevation**=*name*
    **water_elevation_stddev**=*name*
    [**filled_elevation**=*name*]
    **percentile**=*float*
    [**max_stddev**=*float*]
    [**min_size**=*integer*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.hydro.flatten.py input=name water_elevation=name water_elevation_stddev=name percentile=5
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.hydro.flatten.py***",
        **input**,
        **breaklines**=*None*,
        **water_elevation**,
        **water_elevation_stddev**,
        **filled_elevation**=*None*,
        **percentile**=*5*,
        **max_stddev**=*None*,
        **min_size**=*None*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.hydro.flatten.py", input="name", water_elevation="name", water_elevation_stddev="name", percentile=5)
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_hydro_flatten_py*(**input**,
        **breaklines**=*None*,
        **water_elevation**,
        **water_elevation_stddev**,
        **filled_elevation**=*None*,
        **percentile**=*5*,
        **max_stddev**=*None*,
        **min_size**=*None*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_hydro_flatten_py(input="name", water_elevation="name", water_elevation_stddev="name", percentile=5)
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map of binned lidar point elevation  
    **breaklines**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector map of breaklines  
    **water_elevation**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map of derived water elevation  
    &nbsp;&nbsp;&nbsp;&nbsp;Represents single elevation value for each water body  
    **water_elevation_stddev**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map of derived water elevation standard deviation  
    **filled_elevation**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map representing filled digital elevation model  
    **percentile**=*float* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Percentile of elevation to determine water level  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *5*  
    **max_stddev**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum value of standard deviation to fill the DEM  
    **min_size**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum size of areas in map units  
    **-k**  
    &nbsp;&nbsp;&nbsp;&nbsp;Keep intermediate results  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map of binned lidar point elevation  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **breaklines** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector map of breaklines  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **water_elevation** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map of derived water elevation  
    &nbsp;&nbsp;&nbsp;&nbsp;Represents single elevation value for each water body  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **water_elevation_stddev** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map of derived water elevation standard deviation  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **filled_elevation** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map representing filled digital elevation model  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **percentile** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Percentile of elevation to determine water level  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *5*  
    **max_stddev** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum value of standard deviation to fill the DEM  
    **min_size** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum size of areas in map units  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *k*  
    &nbsp;&nbsp;&nbsp;&nbsp;**k**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Keep intermediate results  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map of binned lidar point elevation  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **breaklines** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector map of breaklines  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **water_elevation** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map of derived water elevation  
    &nbsp;&nbsp;&nbsp;&nbsp;Represents single elevation value for each water body  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **water_elevation_stddev** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map of derived water elevation standard deviation  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **filled_elevation** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map representing filled digital elevation model  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **percentile** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Percentile of elevation to determine water level  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *5*  
    **max_stddev** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum value of standard deviation to fill the DEM  
    **min_size** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum size of areas in map units  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *k*  
    &nbsp;&nbsp;&nbsp;&nbsp;**k**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Keep intermediate results  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

The tool derives single elevation value for water bodies based on lidar
data. These values are used for hydro-flattening a digital elevation
model. The **input** raster is expected to represent ground surface
created by binning lidar data (e.g., using *[r.in.pdal](r.in.pdal.md)*)
with averaged ground elevation. Small gaps in the input are expected.
Large gaps are interpreted as water bodies. The minimum size of a water
body can be set with **min\_size** option in map units.

The output **water\_elevation** is a raster map of water bodies where
each water body has a single value representing the water level
elevation derived from the lidar data at the edge of a water body. Since
the elevation varies along the edge, option **percentile** is used to
determine a single value. The variation along the edge can be examined
with the **water\_elevation\_stddev** output representing the standard
deviation of the lidar elevation values along the water body's edge.
Higher deviation suggests problematic areas that need to be further
inspected. The optional output **filled\_elevation** is a raster map of
the input ground surface filled with the computed **water\_elevation**
raster map.

The **breaklines** parameter is an optional input that specifies a
vector map of lines that represent e.g., a break between an impoundment
and downstream river, allowing correct elevation computation.

The **max_stddev** parameter filters areas that have higher standard deviation
than provided limit, these areas are not included in the
**water\_elevation** and **filled\_elevation**.

To keep the intermediate results for inspection, use flag **-k**.

## NOTES

While this tool was designed for water bodies, it can be used for other
purposes, e.g., for filling a gap in digital elevation models caused by
excluding buildings.

This tool does not interpolate gaps in data, rather it derives a single
value for each gap. The result can be used to fill gaps and the tool can
be run on large areas. For actual gap interpolation, which is typically
more computationally intensive, see
*[r.fillnulls](https://grass.osgeo.org/grass-stable/manuals/r.fillnuls.html)*.

## EXAMPLE

We will download a lidar tile with *[r.in.usgs](r.in.usgs.md)* addon,
use
*[r.in.pdal](https://grass.osgeo.org/grass-stable/manuals/r.in.pdal.html)*
to bin the elevation points at 1 meter resolution, and derive elevation
levels for lakes with minimum size of 4000 m^2.

```sh
# select study area and resolution
g.region n=213300 s=211900 w=653900 e=655300 res=1
# download lidar tile into /tmp
r.in.usgs product=lidar output_directory=/tmp title_filter=Phase2 -d
# bin point elevation using ground and road points with reprojection
r.in.pdal input=/tmp/USGS_LPC_NC_Phase2_2014_LA_37_20164902_.laz output=ground -w class_filter=2,13
# convert elevation from feet to meters
r.mapcalc "ground_m = ground * 0.304800609601219"
# derive elevation of water bodies and standard deviation
r.hydro.flatten input=ground_m water_elevation=water_elevation water_elevation_stddev=water_elevation_stddev filled_elevation=filled percentile=10 misize=4000
```

![image-alt](r_hydro_flatten_input.png)
![image-alt](r_hydro_flatten_output_elevation.png)
![image-alt](r_hydro_flatten_output_stddev.png)  
*Figure: Input binned elevation representing ground with gaps (left),
input overlayed with elevation values estimated for gaps and highlighted
with an outline (middle), input overlayed with standard deviation of the
elevation along the edge of the gaps (right).*

## REFERENCE

Method based on workflow
[presented](https://www.youtube.com/watch?v=p9KCfufNYgE) at NC GIS
Conference 2021 by Doug Newcomb.

## SEE ALSO

*[r.in.pdal](https://grass.osgeo.org/grass-stable/manuals/r.in.pdal.html),
[r.in.usgs](r.in.usgs.md),
[r.fillnulls](https://grass.osgeo.org/grass-stable/manuals/r.fillnulls.html)*

## AUTHOR

Anna Petrasova, [NCSU GeoForAll
Lab](https://geospatial.ncsu.edu/geoforall/)

## SOURCE CODE

Available at: [r.hydro.flatten source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/raster/r.hydro.flatten)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/raster/r.hydro.flatten))  
Latest change: Friday Sep 05 16:58:29 2025 in commit [78d5c17](https://github.com/OSGeo/grass-addons/commit/78d5c170fc9965cbd1236cb8f2c90cc2546f9d53)
