---
name: r.stream.channel
description: Calculates local parameters for individual streams.
keywords: [ raster, hydrology, stream network ]
---

# r.stream.channel

Calculates local parameters for individual streams.

=== "Command line"

    **r.stream.channel**
    [**-dlcm**]
    **stream_rast**=*name*
    **direction**=*name*
    **elevation**=*name*
    [**identifier**=*name*]
    [**distance**=*name*]
    [**difference**=*name*]
    [**gradient**=*name*]
    [**curvature**=*name*]
    [**memory**=*integer*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.stream.channel stream_rast=name direction=name elevation=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.stream.channel***",
        **stream_rast**,
        **direction**,
        **elevation**,
        **identifier**=*None*,
        **distance**=*None*,
        **difference**=*None*,
        **gradient**=*None*,
        **curvature**=*None*,
        **memory**=*300*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.stream.channel", stream_rast="name", direction="name", elevation="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_stream_channel*(**stream_rast**,
        **direction**,
        **elevation**,
        **identifier**=*None*,
        **distance**=*None*,
        **difference**=*None*,
        **gradient**=*None*,
        **curvature**=*None*,
        **memory**=*300*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_stream_channel(stream_rast="name", direction="name", elevation="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **stream_rast**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with stream network  
    **direction**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with flow direction  
    **elevation**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input elevation raster map  
    **identifier**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output unique stream identifier raster map  
    **distance**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output init/join/outlet distance raster map  
    **difference**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output elevation init/join/outlet difference raster map  
    **gradient**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output mean init/join/outlet gradient of stream raster map  
    **curvature**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output local stream curvature raster map  
    **memory**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory used in memory swap mode (MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **-d**  
    &nbsp;&nbsp;&nbsp;&nbsp;Calculate parameters from outlet (downstream values)  
    **-l**  
    &nbsp;&nbsp;&nbsp;&nbsp;Calculate local values (for current cell)  
    **-c**  
    &nbsp;&nbsp;&nbsp;&nbsp;Calculate distance in cell count (ignored local)  
    **-m**  
    &nbsp;&nbsp;&nbsp;&nbsp;Use memory swap (operation is slow)  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **stream_rast** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with stream network  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **direction** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with flow direction  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **elevation** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input elevation raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **identifier** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output unique stream identifier raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **distance** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output init/join/outlet distance raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **difference** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output elevation init/join/outlet difference raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **gradient** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output mean init/join/outlet gradient of stream raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **curvature** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output local stream curvature raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory used in memory swap mode (MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *d*, *l*, *c*, *m*  
    &nbsp;&nbsp;&nbsp;&nbsp;**d**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Calculate parameters from outlet (downstream values)  
    &nbsp;&nbsp;&nbsp;&nbsp;**l**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Calculate local values (for current cell)  
    &nbsp;&nbsp;&nbsp;&nbsp;**c**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Calculate distance in cell count (ignored local)  
    &nbsp;&nbsp;&nbsp;&nbsp;**m**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use memory swap (operation is slow)  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **stream_rast** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with stream network  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **direction** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map with flow direction  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **elevation** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input elevation raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **identifier** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output unique stream identifier raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **distance** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output init/join/outlet distance raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **difference** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output elevation init/join/outlet difference raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **gradient** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output mean init/join/outlet gradient of stream raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **curvature** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output local stream curvature raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory used in memory swap mode (MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *d*, *l*, *c*, *m*  
    &nbsp;&nbsp;&nbsp;&nbsp;**d**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Calculate parameters from outlet (downstream values)  
    &nbsp;&nbsp;&nbsp;&nbsp;**l**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Calculate local values (for current cell)  
    &nbsp;&nbsp;&nbsp;&nbsp;**c**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Calculate distance in cell count (ignored local)  
    &nbsp;&nbsp;&nbsp;&nbsp;**m**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use memory swap (operation is slow)  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

The module *r.stream.channel* is prepared to calculate some local
properties of the stream network. It is supplementary module for
*r.stream.order*, and *r.stream.distance* to investigate channel
subsystem. For slope subsystem parameters is *r.stream.slope*. It may
use ordered or unordered network. It calculate parameters for every
segment between it init to outlet/join to the next stream. it also may
calculate parameters between outlet and segment's init. It can calculate
parameters for every orders but best results are for these orders where
order remains unchanged from stream init to natural outlet (Hack and
Horton ordering).

## OPTIONS

- **-d**  
    Calculate downstream distance (from current cell DOWNSTREAM to
    outlet/join). Default is upstream (from current cell upstream to
    init/join.
- **-m**  
    Only for very large data sets. Uses segment library to optimize
    memory consumption during analysis
- **stream\_rast**  
    Stream network: name of input stream map. Map may be ordered
    according to one of the *r.stream.order* ordering systems as well as
    unordered (with original stream identifiers). Because the streams
    network produced by *r.watershed* and *r.stream.extract* may
    slightly differ in detail it is required to use both stream and
    direction map produced by the same module. Non-stream cell values
    must be set to NULL.
- **direction**  
    Flow direction: name of input direction map produced by
    *r.watershed* or *r.stream.extract*. If the *r.stream.extract*
    output map is used, it contains non-NULL values only in places where
    streams are present. NULL cells are ignored; zero and negative
    values are valid direction data if they vary from -8 to 8 (CCW from
    East in steps of 45 degrees). The direction map shall be of integer
    type (CELL). The region resolution and map resolution must be the
    same. Also the *stream\_rast* network map must have the same
    resolution. If resolutions differ the module informs about it and
    stops. Region boundary and maps boundary may be different but it may
    lead to unexpected results.
- **elevation**  
    Elevation: name of input elevation map. Map can be of type CELL,
    FCELL or DCELL. It is not restricted to resolution of region
    settings as streams and direction.
- **distance**  
    Upstream distance of current cell to the init/join. Flag
    modifications:  
    **-d:** downstream distance of current cell to the join/outlet;  
    **-l:** local distance between current cell and next cell. In most
    cases cell resolution and sqrt2 of cell resolution. useful when
    projection is LL or NS and WE resolutions differs. Flag **-d**
    ignored  
    **-c:** distance in cells. Map is written as double. Use *r.mapcalc*
    to convert to integer. Flags **-l** and **-d** ignored.  
- **difference**  
    Upstream elevation difference between current cell to the init/join.
    It we need to calculate parameters different than elevation. If we
    need to calculate different parameters than elevation along streams
    (for example precipitation or so) use necessary map as elevation.
    Flag modifications:  
    **-d:** downstream difference of current cell to the join/outlet;  
    **-l:** local difference between current cell and next cell. With
    flag calculates difference between previous cell and current cell  
    **-c:** Ignored.
- **gradient**  
    Upstream mean gradient between current cell and the init/join. Flag
    modifications:  
    **-d:** downstream mean gradient between current cell and the
    join/outlet;  
    **-l:** local gradient between current cell and next cell. Flag
    **-d** ignored  
    **-c:** Ignored.
- **curvature**  
    Local stream course curvature of current cell. Calculated according
    formula:
    *first\_derivative/(1-second\_derivative<sup>2</sup>)<sup>3/2</sup>*
    Flag modifications:  
    **-d:** ignored;  
    **-l:** Ignored.  
    **-c:** Ignored.
- **identifier**  
    Integer map: In ordered stream network there are more than one
    segment (segment: a part of the network where order remains
    unchanged) with the same order. To identify particular segments (for
    further analysis) every segment receive his unique identifier.

## EXAMPLE

This example shows how to visualise the change of gradient map along the
main stream of the catchment:

```sh
g.region -p -a raster=elevation
r.watershed elevation=elevation threshold=10000 drainage=direction stream=streams
r.stream.order streams=streams direction=direction hack=hack
r.stream.channel streams=hack direction=direction elevation=elevation \
  identifier=stream_identifier distance=stream_distance gradient=stream_gradient

# Eg., 495 is a stream identifier. May be different in different situation
r.mapcalc "stgrad = if(stream_identifier==495,float(stream_gradient),null())"
r.mapcalc "stdist = if(stream_identifier==495,float(stream_distance),null())"

# Use R for plotting
R
library(spgrass6)
r=readRAST6(c("stdist","stgrad"),NODATA=-9999)
p=subset(r@data,!is.na(r@data$dist))
sorted=p[order(p$stdist),]
plot(sorted,stdist~stgrad,type="l")
```

## SEE ALSO

*[r.mapcalc](https://grass.osgeo.org/grass-stable/manuals/r.mapcalc.html),
[r.stream.distance](r.stream.distance.md),
[r.stream.extract](https://grass.osgeo.org/grass-stable/manuals/r.stream.extract.html),
[r.stream.order](r.stream.order.md),
[r.stream.basins](r.stream.basins.md),
[r.stream.segment](r.stream.segment.md),
[r.stream.slope](r.stream.slope.md), [r.stream.snap](r.stream.snap.md),
[r.stream.stats](r.stream.stats.md),
[r.watershed](https://grass.osgeo.org/grass-stable/manuals/r.watershed.html)*

## AUTHOR

Jarek Jasiewicz, Adam Mickiewicz University, Geoecology and
Geoinformation Institute.

## SOURCE CODE

Available at: [r.stream.channel source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/raster/r.stream.channel)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/raster/r.stream.channel))  
Latest change: Friday Feb 21 10:10:05 2025 in commit [7d78fe3](https://github.com/OSGeo/grass-addons/commit/7d78fe34868674c3b6050ba1924e1c5675d155c9)
