---
name: i.feotio2
description: Calculates the FeO or TiO2 contents from the Clementine project Moon data.
keywords: [ imagery, Moon, Clementine, Chandrayaan1, UVVIS, M3, FeO, TiO2, reflectance ]
---

# i.feotio2

Calculates the FeO or TiO2 contents from the Clementine project Moon data.

=== "Command line"

    **i.feotio2**
    [**-t**]
    **band0**=*name*
    **band1**=*name*
    **output**=*name*
    [**param0**=*float*]
    [**param1**=*float*]
    **eqname**=*type*
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    i.feotio2 band0=name band1=name output=name eqname=feolucey2000
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***i.feotio2***",
        **band0**,
        **band1**,
        **output**,
        **param0**=*None*,
        **param1**=*None*,
        **eqname**=*"feolucey2000"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("i.feotio2", band0="name", band1="name", output="name", eqname="feolucey2000")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.i_feotio2*(**band0**,
        **band1**,
        **output**,
        **param0**=*None*,
        **param1**=*None*,
        **eqname**=*"feolucey2000"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.i_feotio2(band0="name", band1="name", output="name", eqname="feolucey2000")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **band0**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;reflectance band at 750 nm (UVVIS 2, M3 9)  
    **band1**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;reflectance band at 950 nm (UVVIS 4 for FeO/OMAT, M3 19) or at 415 nm (UVVIS 1 for TiO2)  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    **param0**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Value of theta (FeO) or y0Ti (TiO2)  
    **param1**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Value of s0Ti (TiO2)  
    **eqname**=*type* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Type of equation  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *feolucey2000, feolawrence2002, feowilcox2005_setparam, feowilcox2005, feozhang2013, omatlucey2000, omatwilcox2005, tio2lucey2000_setparam, tio2lucey2000*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *feolucey2000*  
    &nbsp;&nbsp;&nbsp;&nbsp;**feolucey2000**: FeO (Lucey et al., 2000)  
    &nbsp;&nbsp;&nbsp;&nbsp;**feolawrence2002**: FeO (lawrence et al., 2002)  
    &nbsp;&nbsp;&nbsp;&nbsp;**feowilcox2005_setparam**: FeO (Wilcox et al., 2005) set parameter theta  
    &nbsp;&nbsp;&nbsp;&nbsp;**feowilcox2005**: FeO (Wilcox et al., 2005)  
    &nbsp;&nbsp;&nbsp;&nbsp;**feozhang2013**: FeO (Zhang et al., 2013)  
    &nbsp;&nbsp;&nbsp;&nbsp;**omatlucey2000**: OMAT (Lucey et al., 2000)  
    &nbsp;&nbsp;&nbsp;&nbsp;**omatwilcox2005**: OMAT (Wilcox et al., 2005)  
    &nbsp;&nbsp;&nbsp;&nbsp;**tio2lucey2000_setparam**: TiO2 (Lucey et al., 2000) set parameters  
    &nbsp;&nbsp;&nbsp;&nbsp;**tio2lucey2000**: TiO2 (Lucey et al., 2000)  
    **-t**  
    &nbsp;&nbsp;&nbsp;&nbsp;TiO2 (refl@415, refl@750 & y0Ti, s0Ti)  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **band0** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;reflectance band at 750 nm (UVVIS 2, M3 9)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **band1** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;reflectance band at 950 nm (UVVIS 4 for FeO/OMAT, M3 19) or at 415 nm (UVVIS 1 for TiO2)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **param0** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Value of theta (FeO) or y0Ti (TiO2)  
    **param1** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Value of s0Ti (TiO2)  
    **eqname** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Type of equation  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *type*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *feolucey2000, feolawrence2002, feowilcox2005_setparam, feowilcox2005, feozhang2013, omatlucey2000, omatwilcox2005, tio2lucey2000_setparam, tio2lucey2000*  
    &nbsp;&nbsp;&nbsp;&nbsp;**feolucey2000**: FeO (Lucey et al., 2000)  
    &nbsp;&nbsp;&nbsp;&nbsp;**feolawrence2002**: FeO (lawrence et al., 2002)  
    &nbsp;&nbsp;&nbsp;&nbsp;**feowilcox2005_setparam**: FeO (Wilcox et al., 2005) set parameter theta  
    &nbsp;&nbsp;&nbsp;&nbsp;**feowilcox2005**: FeO (Wilcox et al., 2005)  
    &nbsp;&nbsp;&nbsp;&nbsp;**feozhang2013**: FeO (Zhang et al., 2013)  
    &nbsp;&nbsp;&nbsp;&nbsp;**omatlucey2000**: OMAT (Lucey et al., 2000)  
    &nbsp;&nbsp;&nbsp;&nbsp;**omatwilcox2005**: OMAT (Wilcox et al., 2005)  
    &nbsp;&nbsp;&nbsp;&nbsp;**tio2lucey2000_setparam**: TiO2 (Lucey et al., 2000) set parameters  
    &nbsp;&nbsp;&nbsp;&nbsp;**tio2lucey2000**: TiO2 (Lucey et al., 2000)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *feolucey2000*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *t*  
    &nbsp;&nbsp;&nbsp;&nbsp;**t**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;TiO2 (refl@415, refl@750 & y0Ti, s0Ti)  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **band0** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;reflectance band at 750 nm (UVVIS 2, M3 9)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **band1** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;reflectance band at 950 nm (UVVIS 4 for FeO/OMAT, M3 19) or at 415 nm (UVVIS 1 for TiO2)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **param0** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Value of theta (FeO) or y0Ti (TiO2)  
    **param1** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Value of s0Ti (TiO2)  
    **eqname** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Type of equation  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *type*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *feolucey2000, feolawrence2002, feowilcox2005_setparam, feowilcox2005, feozhang2013, omatlucey2000, omatwilcox2005, tio2lucey2000_setparam, tio2lucey2000*  
    &nbsp;&nbsp;&nbsp;&nbsp;**feolucey2000**: FeO (Lucey et al., 2000)  
    &nbsp;&nbsp;&nbsp;&nbsp;**feolawrence2002**: FeO (lawrence et al., 2002)  
    &nbsp;&nbsp;&nbsp;&nbsp;**feowilcox2005_setparam**: FeO (Wilcox et al., 2005) set parameter theta  
    &nbsp;&nbsp;&nbsp;&nbsp;**feowilcox2005**: FeO (Wilcox et al., 2005)  
    &nbsp;&nbsp;&nbsp;&nbsp;**feozhang2013**: FeO (Zhang et al., 2013)  
    &nbsp;&nbsp;&nbsp;&nbsp;**omatlucey2000**: OMAT (Lucey et al., 2000)  
    &nbsp;&nbsp;&nbsp;&nbsp;**omatwilcox2005**: OMAT (Wilcox et al., 2005)  
    &nbsp;&nbsp;&nbsp;&nbsp;**tio2lucey2000_setparam**: TiO2 (Lucey et al., 2000) set parameters  
    &nbsp;&nbsp;&nbsp;&nbsp;**tio2lucey2000**: TiO2 (Lucey et al., 2000)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *feolucey2000*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *t*  
    &nbsp;&nbsp;&nbsp;&nbsp;**t**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;TiO2 (refl@415, refl@750 & y0Ti, s0Ti)  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*i.feo\_tio2* calculates the FeO or TiO2 contents from Clementine data.
Clementine UVVIS sensor has five bands, UVVIS1 = 415, UVVIS2 = 750,
UVVIS3 = 900, UVVIS4 = 950 and UVVIS5 = 1000 nm.

```text
wt%TiO2 = 3.708 arctan ((R415/R750)-y0Ti)/(R750-s0Ti) (1)

wt%FeO = -137.97 ( R750 sin(theta) + (R950/R750) cos(theta) ) + 57.46   (2)
```

with theta = 1.3885 rad, the 'average slope of the trends in the mare'
from Wilcox et al (2005). [Map-a-Planet
Explorer](https://web.archive.org/web/20161203071556/http://www.mapaplanet.org/explorer/help/index.html)
mentions this set of equations:

```c
double feolucey2000(double uvvis2, double uvvis4){
     //\cite{lucey2000lunar}
     return(17.427*(-atan2f(((uvvis4/uvvis2)-1.19)/(uvvis2-0.08)))-7.565);
}

double feolawrence2002(double uvvis2, double uvvis4){
     //\cite{lawrence2002iron}
     return(5.7*((-0.147+0.372*(-(uvvis4/uvvis2-1.22)/(uvvis2-0.04))+(-0.036)*pow((-(uvvis4/uvvis2-1.22)/(uvvis2-0.04)),2)))+2.15);
}

double feowilcox2005(double uvvis2, double uvvis4){
      //\cite{wilcox2005mapping}
      return(-137.97*((uvvis2*0.9834)+(uvvis4/uvvis2*0.1813))+57.46);
}

double omatlucey2000(double uvvis2, double uvvis4){
     //\cite{lucey2000lunar}
     return(sqrtf(pow((uvvis2-0.08),2)+pow(((uvvis4/uvvis2)-1.19),2)));
}

double omatwilcox2005(double uvvis2, double uvvis4){
     //\cite{wilcox2005mapping}
     return((uvvis2*0.1813)-((uvvis4/uvvis2)*0.9834));
}

double tio2lucey2000(double uvvis1, double uvvis2){
     //\cite{lucey2000lunar}
     return(3.708*pow((atan2f(((uvvis1/uvvis2)-0.42)/(uvvis2-0.0))),5.979));
}
```

## NOTES

Initially created for Clementine data.

## REFERENCES

(1) Lucey, P.G., Blewett, D.T., Jolliff, B.L., 2000. Lunar iron and
titanium abundance algorithms based on final processing of Clementine
ultraviolet-visible images. J. Geophys. Res. 105(E8): 20297-20305. (2)
Wilcox, B.B., Lucey, P.G., Gillis, J.J., 2005. Mapping iron in the lunar
mare: An improved approach. J. Geophys. Res. 110(E11):2156-2202. (3)
Lawrence, DJ and Feldman, WC and Elphic, RC and Little, RC and
Prettyman, TH and Maurice, S and Lucey, PG and Binder, AB, 2002. Iron
abundances on the lunar surface as measured by the Lunar Prospector
gamma-ray and neutron spectrometers. Journal of Geophysical Research:
Planets (1991--2012), 107(E12):13-1.

## AUTHOR

Yann Chemin (B.Sc. student), Birkbeck, University of London.

## SOURCE CODE

Available at: [i.feotio2 source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/imagery/i.feotio2)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/imagery/i.feotio2))  
Latest change: Thursday Mar 20 21:36:57 2025 in commit [7286ecf](https://github.com/OSGeo/grass-addons/commit/7286ecf7af235bfd089fb9b1b82fb383cf95f3fc)
