---
name: r.buildvrt.gdal.py
description: Build GDAL Virtual Rasters (VRT) over GRASS GIS raster maps
keywords: [ raster, virtual, gdal, patch ]
---

# r.buildvrt.gdal.py

Build GDAL Virtual Rasters (VRT) over GRASS GIS raster maps

=== "Command line"

    **r.buildvrt.gdal.py**
    [**-mr**]
    [**input**=*name* [,*name*,...]]
    [**file**=*name*]
    [**vrt_directory**=*name*]
    **output**=*name*
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.buildvrt.gdal.py input=name output=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.buildvrt.gdal.py***",
        **input**=*None*,
        **file**=*None*,
        **vrt_directory**=*None*,
        **output**,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.buildvrt.gdal.py", input="name", output="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_buildvrt_gdal_py*(**input**=*None*,
        **file**=*None*,
        **vrt_directory**=*None*,
        **output**,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_buildvrt_gdal_py(input="name", output="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* [,*name*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map(s)  
    **file**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input file  
    **vrt_directory**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Directory to store GDAL VRT files in. Default is: $GISDBASE/$PROJECT/$MAPSET/gdal  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    **-m**  
    &nbsp;&nbsp;&nbsp;&nbsp;Read data range from metadata  
    &nbsp;&nbsp;&nbsp;&nbsp;WARNING: metadata are sometimes approximations with wrong data range  
    **-r**  
    &nbsp;&nbsp;&nbsp;&nbsp;Create fast link without data range  
    &nbsp;&nbsp;&nbsp;&nbsp;WARNING: some modules do not work correctly without known data range  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map(s)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **file** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input file  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **vrt_directory** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Directory to store GDAL VRT files in. Default is: $GISDBASE/$PROJECT/$MAPSET/gdal  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dir, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *m*, *r*  
    &nbsp;&nbsp;&nbsp;&nbsp;**m**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Read data range from metadata  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;WARNING: metadata are sometimes approximations with wrong data range  
    &nbsp;&nbsp;&nbsp;&nbsp;**r**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Create fast link without data range  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;WARNING: some modules do not work correctly without known data range  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map(s)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **file** : str | io.StringIO, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input file  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **vrt_directory** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Directory to store GDAL VRT files in. Default is: $GISDBASE/$PROJECT/$MAPSET/gdal  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, dir, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *m*, *r*  
    &nbsp;&nbsp;&nbsp;&nbsp;**m**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Read data range from metadata  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;WARNING: metadata are sometimes approximations with wrong data range  
    &nbsp;&nbsp;&nbsp;&nbsp;**r**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Create fast link without data range  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;WARNING: some modules do not work correctly without known data range  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.buildvrt.gdal* builds GDAL virtual rasters over GRASS GIS raster maps
and links them to the mapset with *r.external*. The module is written as
a workaround for a limitation in GRASS GIS Virtual Rasters (VRT) format
with GDAL-linked raster maps (through *r.external* / *r.external.out*.
In that case GRASS GIS Virtual Rasters currently show performance
issues. See: [\#4345](https://github.com/OSGeo/grass/issues/4345)

For the resulting maps GDAL VRT text files are created either in a
directory named "gdal" in the current mapset or in a user-defined
**vrt\_directory**. Those files are not removed when the raster map is
removed and the user is responsible for removing them when needed.

## REQUIREMENTS

*r.buildvrt.gdal* uses the Python bindings for
[GDAL](https://pypi.org/project/GDAL) and requires the GDAL-GRASS driver
to include raster maps in native GRASS format in GDAL VRTs.

## EXAMPLES

```sh
# Create external example data
regs='s,0,1000
n,500,1500'

eval `g.gisenv`
external_path="${GISDBASE}/${LOCATION}/${MAPSET}/.tmp/vrt"
mkdir -p "$external_path"
for reg in $regs
do
  r=$(echo $reg | cut -f1 -d",")
  s=$(echo $reg | cut -f2 -d",")
  n=$(echo $reg | cut -f3 -d",")

  g.region -g n=$n s=$s w=0 e=1000 res=1
  r.external.out format=GTiff options="compress=LZW,PREDICTOR=3" \
    directory="$external_path"
  r.mapcalc --o --v expression="${r}_${s}_gtiff_ntfs=float(x()*y())"
done

# Run performance tests
g.region -g n=1500 s=0 w=0 e=1000 res=1
format_type=gtiff_ntfs
rmaps=$(g.list type=raster pattern="*_*_${format_type}", sep=",")

# Using GRASS GIS VRT
r.buildvrt --o --v input="$rmaps" output=vrt_${format_type}
time r.univar map=vrt_${format_type}

# Using GDAL VRT
r.buildvrt.gdal --o --v input="$rmaps" output=vrt_${format_type}_gdal
time r.univar map=vrt_${format_type}_gdal
```

## SEE ALSO

*[r.buildvrt](https://grass.osgeo.org/grass-stable/manuals/r.buildvrt.html),
[r.patch](https://grass.osgeo.org/grass-stable/manuals/r.patch.html),
[r.external](https://grass.osgeo.org/grass-stable/manuals/r.external.html),
[r.external.out](https://grass.osgeo.org/grass-stable/manuals/r.external.out.html)*

## AUTHORS

Stefan Blumentrath

## SOURCE CODE

Available at: [r.buildvrt.gdal source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/raster/r.buildvrt.gdal)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/raster/r.buildvrt.gdal))  
Latest change: Thursday Feb 20 13:02:26 2025 in commit [53de819](https://github.com/OSGeo/grass-addons/commit/53de8196a10ba5a8a9121898ce87861d227137e3)
