---
name: r.gwr
description: Calculates geographically weighted regression from raster maps.
keywords: [ raster, statistics, regression ]
---

# r.gwr

Calculates geographically weighted regression from raster maps.

=== "Command line"

    **r.gwr**
    [**-ge**]
    **mapx**=*name* [,*name*,...]
    **mapy**=*name*
    [**mask**=*name*]
    [**residuals**=*name*]
    [**estimates**=*name*]
    [**coefficients**=*string*]
    [**output**=*name*]
    [**kernel**=*string*]
    [**bandwidth**=*integer*]
    [**vf**=*integer*]
    [**npoints**=*integer*]
    [**memory**=*integer*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.gwr mapx=name mapy=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.gwr***",
        **mapx**,
        **mapy**,
        **mask**=*None*,
        **residuals**=*None*,
        **estimates**=*None*,
        **coefficients**=*None*,
        **output**=*None*,
        **kernel**=*"gauss"*,
        **bandwidth**=*10*,
        **vf**=*1*,
        **npoints**=*0*,
        **memory**=*300*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.gwr", mapx="name", mapy="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_gwr*(**mapx**,
        **mapy**,
        **mask**=*None*,
        **residuals**=*None*,
        **estimates**=*None*,
        **coefficients**=*None*,
        **output**=*None*,
        **kernel**=*"gauss"*,
        **bandwidth**=*10*,
        **vf**=*1*,
        **npoints**=*0*,
        **memory**=*300*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_gwr(mapx="name", mapy="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **mapx**=*name* [,*name*,...] **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Map(s) with X variables  
    **mapy**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Map with Y variable  
    **mask**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map to use for masking  
    &nbsp;&nbsp;&nbsp;&nbsp;Only cells that are not NULL and not zero are processed  
    **residuals**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map to store residuals  
    **estimates**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map to store estimates  
    **coefficients**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Prefix for maps to store coefficients  
    **output**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;ASCII file for storing regression coefficients (output to screen if file not specified).  
    **kernel**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Weighing kernel function.  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *gauss, epanechnikov, bisquare, tricubic*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *gauss*  
    **bandwidth**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Bandwidth of the weighing kernel.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *10*  
    **vf**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Variance factor for Gaussian kernel: variance = bandwith / factor.  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1, 2, 4, 8*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **npoints**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of points for adaptive bandwidth  
    &nbsp;&nbsp;&nbsp;&nbsp;If 0, fixed bandwidth is used  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **memory**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Memory in MB for adaptive bandwidth  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **-g**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print in shell script style  
    **-e**  
    &nbsp;&nbsp;&nbsp;&nbsp;Estimate optimal bandwidth  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **mapx** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map(s) with X variables  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **mapy** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map with Y variable  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **mask** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map to use for masking  
    &nbsp;&nbsp;&nbsp;&nbsp;Only cells that are not NULL and not zero are processed  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **residuals** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map to store residuals  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **estimates** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map to store estimates  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **coefficients** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Prefix for maps to store coefficients  
    **output** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;ASCII file for storing regression coefficients (output to screen if file not specified).  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    **kernel** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Weighing kernel function.  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *gauss, epanechnikov, bisquare, tricubic*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *gauss*  
    **bandwidth** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Bandwidth of the weighing kernel.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *10*  
    **vf** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Variance factor for Gaussian kernel: variance = bandwith / factor.  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1, 2, 4, 8*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **npoints** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of points for adaptive bandwidth  
    &nbsp;&nbsp;&nbsp;&nbsp;If 0, fixed bandwidth is used  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Memory in MB for adaptive bandwidth  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *g*, *e*  
    &nbsp;&nbsp;&nbsp;&nbsp;**g**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print in shell script style  
    &nbsp;&nbsp;&nbsp;&nbsp;**e**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Estimate optimal bandwidth  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **mapx** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map(s) with X variables  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **mapy** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map with Y variable  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **mask** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map to use for masking  
    &nbsp;&nbsp;&nbsp;&nbsp;Only cells that are not NULL and not zero are processed  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **residuals** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map to store residuals  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **estimates** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Map to store estimates  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **coefficients** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Prefix for maps to store coefficients  
    **output** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;ASCII file for storing regression coefficients (output to screen if file not specified).  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    **kernel** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Weighing kernel function.  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *gauss, epanechnikov, bisquare, tricubic*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *gauss*  
    **bandwidth** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Bandwidth of the weighing kernel.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *10*  
    **vf** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Variance factor for Gaussian kernel: variance = bandwith / factor.  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1, 2, 4, 8*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **npoints** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of points for adaptive bandwidth  
    &nbsp;&nbsp;&nbsp;&nbsp;If 0, fixed bandwidth is used  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Memory in MB for adaptive bandwidth  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *g*, *e*  
    &nbsp;&nbsp;&nbsp;&nbsp;**g**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Print in shell script style  
    &nbsp;&nbsp;&nbsp;&nbsp;**e**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Estimate optimal bandwidth  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.gwr* calculates a geographically weighted multiple linear regression
from raster maps, according to the formula

```text
Y = b0 + sum(bi*Xi) + E
```

where

```text
X = {X1, X2, ..., Xm}
m = number of explaining variables
Y = {y1, y2, ..., yn}
Xi = {xi1, xi2, ..., xin}
E = {e1, e2, ..., en}
n = number of observations (cases)
```

In R notation:

```R
Y ~ sum(bi*Xi)
b0 is the intercept, X0 is set to 1
```

The β coefficients are localized, i.e. determined for each cell
individually. These β coefficients are the most important output of
*r.gwr*. Spatial patterns and localized outliers in these coefficients
can reveal details of the relation of Y to X. Outliers in the β
coefficients can also be caused by a small bandwidth and can be removed
by increasing the bandwidth.

Geographically weighted regressions should be used as a diagnostic tool
and not as an interpolation method. If a geographically weighted
regression provides a higher R squared than the corresponding global
regression, then a crucial predictor is missing in the model. If that
missing predictor can not be estimated or is supposed to behave
randomly, a geographically weighted regression might be used for
interpolation, but the result, in particular the variation of the β
coefficients needs to be judged according to prior assumptions. See also
the manual and the examples of the R package
[spgwr](http://cran.rstudio.com/web/packages/spgwr/index.html).

*r.gwr* is designed for large datasets that can not be processed in R. A
p value is therefore not provided, because even very small, meaningless
effects will become significant with a large number of cells. Instead it
is recommended to judge by the amount of variance explained (R squared
for a given variable) and the gain in AIC (AIC without a given variable
minus AIC global must be positive) whether the inclusion of a given
explaining variable in the model is justified.

### The explaining variables

R squared for each explaining variable represents the additional amount
of explained variance when including this variable compared to when
excluding this variable, that is, this amount of variance is explained
by the current explaining variable after taking into consideration all
the other explaining variables.

The F score for each explaining variable allows to test if the inclusion
of this variable significantly increases the explaining power of the
model, relative to the global model excluding this explaining variable.
That means that the F value for a given explaining variable is only
identical to the F value of the R-function *summary.aov* if the given
explaining variable is the last variable in the R-formula. While R
successively includes one variable after another in the order specified
by the formula and at each step calculates the F value expressing the
gain by including the current variable in addition to the previous
variables, *r.gwr* calculates the F-value expressing the gain by
including the current variable in addition to all other variables, not
only the previous variables.

### Bandwidth

The bandwidth is the crucial parameter for geographically weighed
regressions. A too small bandwidth will essentially use the weighed
average, any predictors are mostly ignored. A too large bandwidth will
produce results similar to a global regression, and spatial
non-stationarity can not be explored.

### Adaptive bandwidth

Instead of using a fixed bandwidth (search radius for each cell), an
adaptive bandwidth can be used by specifying the number of points to be
used for each local regression with the *npoints* option. The module
will find the nearest *npoints* points for each cell, adapt the bandwith
accordingly and then calculate a local weighted regression.

### Kernel functions

The kernel function has little influence on the result, much more
important is the bandwidth. Available kernel funtions to calculate
weights are

- **Epanechnikov**  
    w = 1 - d / bw
- **Bisquare (Quartic)**  
    w = (1 - (d / bw)<sup>2</sup>)<sup>2</sup>
- **Tricubic**  
    w = (1 - (d / bw)<sup>3</sup>)<sup>3</sup>
- **Gaussian**  
    w = exp(-0.5 \* (d / bw)<sup>2</sup>)

with  
w = weight for current cell  
d = distance to the current cell  
bw = bandwidth

### Masking

A *mask* map can be provided (e.g. with **r.mask**) to restrict LWR to
those cells where the mask map is not NULL and not 0 (zero).

## REFERENCES

Brunsdon, C., Fotheringham, A.S., and Charlton, M.E., 1996,
Geographically Weighted Regression: A Method for Exploring Spatial
Nonstationarity, Geographical Analysis, 28(4), 281- 298  
Fotheringham, A.S., Brunsdon, C., and Charlton, M.E., 2002,
Geographically Weighted Regression: The Analysis of Spatially Varying
Relationships, Chichester: Wiley.  

## SEE ALSO

<https://geoinformatics.wp.st-andrews.ac.uk/gwr/>  
<http://gwr.nuim.ie/>  
R package [spgwr](https://cran.r-project.org/package=spgwr)

## AUTHOR

Markus Metz

## SOURCE CODE

Available at: [r.gwr source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/raster/r.gwr)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/raster/r.gwr))  
Latest change: Sunday Jan 11 13:16:41 2026 in commit [4e21106](https://github.com/OSGeo/grass-addons/commit/4e21106a15350697580ad96e27424a17791a45cb)
