---
name: r.learn.predict.py
description: Apply a fitted scikit-learn estimator to rasters in a GRASS GIS imagery group.
keywords: [ raster, classification, regression, machine learning, scikit-learn, prediction ]
---

# r.learn.predict.py

Apply a fitted scikit-learn estimator to rasters in a GRASS GIS imagery group.

=== "Command line"

    **r.learn.predict.py**
    [**-pz**]
    **group**=*name*
    **load_model**=*name*
    **output**=*name*
    [**chunksize**=*integer*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.learn.predict.py group=name load_model=name output=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.learn.predict.py***",
        **group**,
        **load_model**,
        **output**,
        **chunksize**=*100000*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.learn.predict.py", group="name", load_model="name", output="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_learn_predict_py*(**group**,
        **load_model**,
        **output**,
        **chunksize**=*100000*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_learn_predict_py(group="name", load_model="name", output="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **group**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Group of raster layers used for prediction  
    &nbsp;&nbsp;&nbsp;&nbsp;GRASS imagery group of raster maps representing feature variables to be used in the machine learning model  
    **load_model**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Load model from file  
    &nbsp;&nbsp;&nbsp;&nbsp;File representing pickled scikit-learn estimator model  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output Map  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster layer name to store result from classification or regression model. The name will also used as a perfix if class probabilities or intermediate of cross-validation results are ordered as maps.  
    **chunksize**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of pixels to pass to the prediction method  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of pixels to pass to the prediction method. GRASS GIS reads raster by-row so chunksize is rounded down based on the number of columns  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *100000*  
    **-p**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output class membership probabilities  
    &nbsp;&nbsp;&nbsp;&nbsp;A raster layer is created for each class. For the case of a binary classification, only the positive (maximum) class is output  
    **-z**  
    &nbsp;&nbsp;&nbsp;&nbsp;Only predict class probabilities  
  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **group** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Group of raster layers used for prediction  
    &nbsp;&nbsp;&nbsp;&nbsp;GRASS imagery group of raster maps representing feature variables to be used in the machine learning model  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, group, *name*  
    **load_model** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Load model from file  
    &nbsp;&nbsp;&nbsp;&nbsp;File representing pickled scikit-learn estimator model  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output Map  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster layer name to store result from classification or regression model. The name will also used as a perfix if class probabilities or intermediate of cross-validation results are ordered as maps.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **chunksize** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of pixels to pass to the prediction method  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of pixels to pass to the prediction method. GRASS GIS reads raster by-row so chunksize is rounded down based on the number of columns  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *100000*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *p*, *z*  
    &nbsp;&nbsp;&nbsp;&nbsp;**p**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Output class membership probabilities  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;A raster layer is created for each class. For the case of a binary classification, only the positive (maximum) class is output  
    &nbsp;&nbsp;&nbsp;&nbsp;**z**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Only predict class probabilities  
  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **group** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Group of raster layers used for prediction  
    &nbsp;&nbsp;&nbsp;&nbsp;GRASS imagery group of raster maps representing feature variables to be used in the machine learning model  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, group, *name*  
    **load_model** : str | io.StringIO, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Load model from file  
    &nbsp;&nbsp;&nbsp;&nbsp;File representing pickled scikit-learn estimator model  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output Map  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster layer name to store result from classification or regression model. The name will also used as a perfix if class probabilities or intermediate of cross-validation results are ordered as maps.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **chunksize** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of pixels to pass to the prediction method  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of pixels to pass to the prediction method. GRASS GIS reads raster by-row so chunksize is rounded down based on the number of columns  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *100000*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *p*, *z*  
    &nbsp;&nbsp;&nbsp;&nbsp;**p**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Output class membership probabilities  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;A raster layer is created for each class. For the case of a binary classification, only the positive (maximum) class is output  
    &nbsp;&nbsp;&nbsp;&nbsp;**z**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Only predict class probabilities  
  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.learn.predict* performs the prediction phase of a machine learning
workflow. The user is required to load a prefitted scikit-learn
estimator using the *load\_model* parameter, which can be developed
using the *r.learn.train* module, or can represent any fitted
scikit-learn compatible estimator that is pickled to a file. The GRASS
GIS imagery group to apply the model is set using the *group* parameter.

## NOTES

*r.learn.predict* is designed to keep system memory requirements
relatively low. For this purpose, the rasters are read from the disk
row-by-row, using the RasterRow method in PyGRASS. This however does not
represent an efficient volume of data to pass to the classifiers, which
are mostly multithreaded. Instead, groups of rows as passed to the
estimator. The *chunksize* parameter represents the maximum memory size
(in MB) for each of these blocks of data. Note that the module will
consume more memory than this, especially if the estimator model was
trained using multiple cores.

## EXAMPLE

Here we are going to use the GRASS GIS sample North Carolina data set as
a basis to perform a landsat classification. We are going to classify a
Landsat 7 scene from 2000, using training information from an older
(1996) land cover dataset.

Landsat 7 (2000) bands 7,4,2 color composite example:

![image-alt](lsat7_2000_b742.png)

Note that this example must be run in the "landsat" mapset of the North
Carolina sample data set location.

First, we are going to generate some training pixels from an older
(1996) land cover classification:

```sh
g.region raster=landclass96 -p
r.random input=landclass96 npoints=1000 raster=training_pixels
```

Then we can use these training pixels to perform a classification on the
more recently obtained landsat 7 image:

```sh
# train a random forest classification model using r.learn.train
r.learn.train group=lsat7_2000 training_map=training_pixels \
  model_name=RandomForestClassifier n_estimators=500 save_model=rf_model.gz

# perform prediction using r.learn.predict
r.learn.predict group=lsat7_2000 load_model=rf_model.gz output=rf_classification

# check raster categories - they are automatically applied to the classification output
r.category rf_classification

# copy color scheme from landclass training map to result
r.colors rf_classification raster=training_pixels
```

Random forest classification result:

![image-alt](rfclassification.png)

## SEE ALSO

[r.learn.ml2](r.learn.ml2.md) (overview),
[r.learn.train](r.learn.train.md)

## REFERENCES

Scikit-learn: Machine Learning in Python, Pedregosa et al., JMLR 12, pp.
2825-2830, 2011.

## AUTHOR

Steven Pawley

## SOURCE CODE

Available at: [r.learn.predict source code](https://github.com/OSGeo/grass-addons/tree/grass8/raster/r.learn.ml2/r.learn.predict)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/raster/r.learn.ml2/r.learn.predict))  
Latest change: Thursday Feb 20 13:02:26 2025 in commit [53de819](https://github.com/OSGeo/grass-addons/commit/53de8196a10ba5a8a9121898ce87861d227137e3)
