---
name: r.sim.terrain.py
description: Dynamic landscape evolution model
keywords: [ raster, terrain, landscape, evolution, parallel ]
---

# r.sim.terrain.py

Dynamic landscape evolution model

=== "Command line"

    **r.sim.terrain.py**
    [**-f**]
    **elevation**=*name*
    **runs**=*string*
    **mode**=*string*
    [**rain_intensity**=*integer*]
    [**rain_duration**=*integer*]
    [**precipitation**=*name*]
    [**k_factor**=*name*]
    [**k_factor_value**=*float*]
    [**c_factor**=*name*]
    [**c_factor_value**=*float*]
    [**m**=*float*]
    [**n**=*float*]
    [**walkers**=*integer*]
    [**runoff**=*name*]
    [**runoff_value**=*float*]
    [**mannings**=*name*]
    [**mannings_value**=*float*]
    [**detachment**=*name*]
    [**detachment_value**=*float*]
    [**transport**=*name*]
    [**transport_value**=*float*]
    [**shearstress**=*name*]
    [**shearstress_value**=*float*]
    [**density**=*name*]
    [**density_value**=*float*]
    [**mass**=*name*]
    [**mass_value**=*float*]
    [**grav_diffusion**=*float*]
    [**erdepmin**=*float*]
    [**erdepmax**=*float*]
    **start**=*string*
    **rain_interval**=*integer*
    **temporaltype**=*name*
    [**threads**=*integer*]
    **elevation_timeseries**=*name*
    [**depth_timeseries**=*name*]
    [**erdep_timeseries**=*name*]
    [**flux_timeseries**=*name*]
    [**difference_timeseries**=*name*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.sim.terrain.py elevation=name runs=event mode=simwe_mode start=2016-01-01 00:00:00 rain_interval=1 temporaltype=absolute elevation_timeseries=elevation_timeseries
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.sim.terrain.py***",
        **elevation**,
        **runs**=*"event"*,
        **mode**=*"simwe_mode"*,
        **rain_intensity**=*50*,
        **rain_duration**=*60*,
        **precipitation**=*None*,
        **k_factor**=*None*,
        **k_factor_value**=*0.25*,
        **c_factor**=*None*,
        **c_factor_value**=*0.1*,
        **m**=*1.5*,
        **n**=*1.2*,
        **walkers**=*1000000*,
        **runoff**=*None*,
        **runoff_value**=*0.35*,
        **mannings**=*None*,
        **mannings_value**=*0.04*,
        **detachment**=*None*,
        **detachment_value**=*0.01*,
        **transport**=*None*,
        **transport_value**=*0.01*,
        **shearstress**=*None*,
        **shearstress_value**=*0.0*,
        **density**=*None*,
        **density_value**=*1.4*,
        **mass**=*None*,
        **mass_value**=*116.*,
        **grav_diffusion**=*0.1*,
        **erdepmin**=*-0.5*,
        **erdepmax**=*0.5*,
        **start**=*"2016-01-01 00:00:00"*,
        **rain_interval**=*1*,
        **temporaltype**=*"absolute"*,
        **threads**=*1*,
        **elevation_timeseries**=*"elevation_timeseries"*,
        **depth_timeseries**=*"depth_timeseries"*,
        **erdep_timeseries**=*"erdep_timeseries"*,
        **flux_timeseries**=*"flux_timeseries"*,
        **difference_timeseries**=*"difference_timeseries"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.sim.terrain.py", elevation="name", runs="event", mode="simwe_mode", start="2016-01-01 00:00:00", rain_interval=1, temporaltype="absolute", elevation_timeseries="elevation_timeseries")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_sim_terrain_py*(**elevation**,
        **runs**=*"event"*,
        **mode**=*"simwe_mode"*,
        **rain_intensity**=*50*,
        **rain_duration**=*60*,
        **precipitation**=*None*,
        **k_factor**=*None*,
        **k_factor_value**=*0.25*,
        **c_factor**=*None*,
        **c_factor_value**=*0.1*,
        **m**=*1.5*,
        **n**=*1.2*,
        **walkers**=*1000000*,
        **runoff**=*None*,
        **runoff_value**=*0.35*,
        **mannings**=*None*,
        **mannings_value**=*0.04*,
        **detachment**=*None*,
        **detachment_value**=*0.01*,
        **transport**=*None*,
        **transport_value**=*0.01*,
        **shearstress**=*None*,
        **shearstress_value**=*0.0*,
        **density**=*None*,
        **density_value**=*1.4*,
        **mass**=*None*,
        **mass_value**=*116.*,
        **grav_diffusion**=*0.1*,
        **erdepmin**=*-0.5*,
        **erdepmax**=*0.5*,
        **start**=*"2016-01-01 00:00:00"*,
        **rain_interval**=*1*,
        **temporaltype**=*"absolute"*,
        **threads**=*1*,
        **elevation_timeseries**=*"elevation_timeseries"*,
        **depth_timeseries**=*"depth_timeseries"*,
        **erdep_timeseries**=*"erdep_timeseries"*,
        **flux_timeseries**=*"flux_timeseries"*,
        **difference_timeseries**=*"difference_timeseries"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_sim_terrain_py(elevation="name", runs="event", mode="simwe_mode", start="2016-01-01 00:00:00", rain_interval=1, temporaltype="absolute", elevation_timeseries="elevation_timeseries")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **elevation**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input elevation raster map  
    **runs**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Run for a single rainfall event or a series of events  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *event, series*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *event*  
    &nbsp;&nbsp;&nbsp;&nbsp;**event**: single rainfall event  
    &nbsp;&nbsp;&nbsp;&nbsp;**series**: series of rainfall events  
    **mode**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;SIMWE erosion deposition, USPED transport limited, or RUSLE 3D detachment limited mode  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *simwe_mode, usped_mode, rusle_mode*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *simwe_mode*  
    &nbsp;&nbsp;&nbsp;&nbsp;**simwe_mode**: SIMWE erosion deposition mode  
    &nbsp;&nbsp;&nbsp;&nbsp;**usped_mode**: USPED transport limited mode  
    &nbsp;&nbsp;&nbsp;&nbsp;**rusle_mode**: RUSLE 3D detachment limited mode  
    **rain_intensity**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Rainfall intensity in mm/hr  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *50*  
    **rain_duration**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Total duration of storm event in minutes  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *60*  
    **precipitation**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Precipitation file  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input precipitation file  
    **k_factor**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;K factor  
    &nbsp;&nbsp;&nbsp;&nbsp;Soil erodibility factor  
    **k_factor_value**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;K factor constant  
    &nbsp;&nbsp;&nbsp;&nbsp;Soil erodibility constant  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.25*  
    **c_factor**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;C factor  
    &nbsp;&nbsp;&nbsp;&nbsp;Land cover factor  
    **c_factor_value**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;C factor constant  
    &nbsp;&nbsp;&nbsp;&nbsp;Land cover constant  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.1*  
    **m**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Water flow exponent  
    &nbsp;&nbsp;&nbsp;&nbsp;Water flow exponent  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.5*  
    **n**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Slope exponent  
    &nbsp;&nbsp;&nbsp;&nbsp;Slope exponent  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.2*  
    **walkers**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of walkers (max = 7000000)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1000000*  
    **runoff**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Runoff coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Runoff coefficient (0.6 for bare earth, 0.35 for grass or crops, 0.5 for shrubs and trees, 0.25 for forest, 0.95 for roads)  
    **runoff_value**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Runoff coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Runoff coefficient (0.6 for bare earth, 0.35 for grass or crops, 0.5 for shrubs and trees, 0.25 for forest, 0.95 for roads)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.35*  
    **mannings**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Manning's roughness coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Manning's roughness coefficient  
    **mannings_value**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Manning's roughness coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Manning's roughness coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.04*  
    **detachment**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Detachment coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Detachment coefficient  
    **detachment_value**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Detachment coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Detachment coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.01*  
    **transport**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Transport coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Transport coefficient  
    **transport_value**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Transport coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Transport coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.01*  
    **shearstress**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Shear stress coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Shear stress coefficient  
    **shearstress_value**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Shear stress coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Shear stress coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.0*  
    **density**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Sediment mass density  
    &nbsp;&nbsp;&nbsp;&nbsp;Sediment mass density in g/cm^3  
    **density_value**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Sediment mass density  
    &nbsp;&nbsp;&nbsp;&nbsp;Sediment mass density in g/cm^3  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.4*  
    **mass**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Mass of sediment per unit area  
    &nbsp;&nbsp;&nbsp;&nbsp;Mass of sediment per unit area in kg/m^2  
    **mass_value**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Mass of sediment per unit area  
    &nbsp;&nbsp;&nbsp;&nbsp;Mass of sediment per unit area in kg/m^2  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *116.*  
    **grav_diffusion**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Gravitational diffusion coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Gravitational diffusion coefficient in m^2/s  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.1*  
    **erdepmin**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum values for erosion-deposition  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum values for erosion-deposition in kg/m^2s  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-0.5*  
    **erdepmax**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum values for erosion-deposition  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum values for erosion-deposition in kg/m^2s  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.5*  
    **start**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Start time in year-month-day hour:minute:second format  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2016-01-01 00:00:00*  
    **rain_interval**=*integer* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Time interval between evolution events in minutes  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **temporaltype**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;The temporal type of the space time dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *absolute, relative*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *absolute*  
    **threads**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of threads for multiprocessing  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **elevation_timeseries**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *elevation_timeseries*  
    **depth_timeseries**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *depth_timeseries*  
    **erdep_timeseries**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *erdep_timeseries*  
    **flux_timeseries**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *flux_timeseries*  
    **difference_timeseries**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *difference_timeseries*  
    **-f**  
    &nbsp;&nbsp;&nbsp;&nbsp;Fill depressions  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **elevation** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input elevation raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **runs** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Run for a single rainfall event or a series of events  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *event, series*  
    &nbsp;&nbsp;&nbsp;&nbsp;**event**: single rainfall event  
    &nbsp;&nbsp;&nbsp;&nbsp;**series**: series of rainfall events  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *event*  
    **mode** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;SIMWE erosion deposition, USPED transport limited, or RUSLE 3D detachment limited mode  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *simwe_mode, usped_mode, rusle_mode*  
    &nbsp;&nbsp;&nbsp;&nbsp;**simwe_mode**: SIMWE erosion deposition mode  
    &nbsp;&nbsp;&nbsp;&nbsp;**usped_mode**: USPED transport limited mode  
    &nbsp;&nbsp;&nbsp;&nbsp;**rusle_mode**: RUSLE 3D detachment limited mode  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *simwe_mode*  
    **rain_intensity** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Rainfall intensity in mm/hr  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *50*  
    **rain_duration** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Total duration of storm event in minutes  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *60*  
    **precipitation** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Precipitation file  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input precipitation file  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **k_factor** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;K factor  
    &nbsp;&nbsp;&nbsp;&nbsp;Soil erodibility factor  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **k_factor_value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;K factor constant  
    &nbsp;&nbsp;&nbsp;&nbsp;Soil erodibility constant  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.25*  
    **c_factor** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;C factor  
    &nbsp;&nbsp;&nbsp;&nbsp;Land cover factor  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **c_factor_value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;C factor constant  
    &nbsp;&nbsp;&nbsp;&nbsp;Land cover constant  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.1*  
    **m** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Water flow exponent  
    &nbsp;&nbsp;&nbsp;&nbsp;Water flow exponent  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.5*  
    **n** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Slope exponent  
    &nbsp;&nbsp;&nbsp;&nbsp;Slope exponent  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.2*  
    **walkers** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of walkers (max = 7000000)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1000000*  
    **runoff** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Runoff coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Runoff coefficient (0.6 for bare earth, 0.35 for grass or crops, 0.5 for shrubs and trees, 0.25 for forest, 0.95 for roads)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **runoff_value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Runoff coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Runoff coefficient (0.6 for bare earth, 0.35 for grass or crops, 0.5 for shrubs and trees, 0.25 for forest, 0.95 for roads)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.35*  
    **mannings** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Manning's roughness coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Manning's roughness coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **mannings_value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Manning's roughness coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Manning's roughness coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.04*  
    **detachment** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Detachment coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Detachment coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **detachment_value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Detachment coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Detachment coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.01*  
    **transport** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Transport coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Transport coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **transport_value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Transport coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Transport coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.01*  
    **shearstress** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Shear stress coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Shear stress coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **shearstress_value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Shear stress coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Shear stress coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.0*  
    **density** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Sediment mass density  
    &nbsp;&nbsp;&nbsp;&nbsp;Sediment mass density in g/cm^3  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **density_value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Sediment mass density  
    &nbsp;&nbsp;&nbsp;&nbsp;Sediment mass density in g/cm^3  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.4*  
    **mass** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Mass of sediment per unit area  
    &nbsp;&nbsp;&nbsp;&nbsp;Mass of sediment per unit area in kg/m^2  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **mass_value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Mass of sediment per unit area  
    &nbsp;&nbsp;&nbsp;&nbsp;Mass of sediment per unit area in kg/m^2  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *116.*  
    **grav_diffusion** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Gravitational diffusion coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Gravitational diffusion coefficient in m^2/s  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.1*  
    **erdepmin** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum values for erosion-deposition  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum values for erosion-deposition in kg/m^2s  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-0.5*  
    **erdepmax** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum values for erosion-deposition  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum values for erosion-deposition in kg/m^2s  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.5*  
    **start** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Start time in year-month-day hour:minute:second format  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2016-01-01 00:00:00*  
    **rain_interval** : int, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Time interval between evolution events in minutes  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **temporaltype** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The temporal type of the space time dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *absolute, relative*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *absolute*  
    **threads** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of threads for multiprocessing  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **elevation_timeseries** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, strds, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *elevation_timeseries*  
    **depth_timeseries** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, strds, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *depth_timeseries*  
    **erdep_timeseries** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, strds, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *erdep_timeseries*  
    **flux_timeseries** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, strds, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *flux_timeseries*  
    **difference_timeseries** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, strds, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *difference_timeseries*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *f*  
    &nbsp;&nbsp;&nbsp;&nbsp;**f**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Fill depressions  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **elevation** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input elevation raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **runs** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Run for a single rainfall event or a series of events  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *event, series*  
    &nbsp;&nbsp;&nbsp;&nbsp;**event**: single rainfall event  
    &nbsp;&nbsp;&nbsp;&nbsp;**series**: series of rainfall events  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *event*  
    **mode** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;SIMWE erosion deposition, USPED transport limited, or RUSLE 3D detachment limited mode  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *simwe_mode, usped_mode, rusle_mode*  
    &nbsp;&nbsp;&nbsp;&nbsp;**simwe_mode**: SIMWE erosion deposition mode  
    &nbsp;&nbsp;&nbsp;&nbsp;**usped_mode**: USPED transport limited mode  
    &nbsp;&nbsp;&nbsp;&nbsp;**rusle_mode**: RUSLE 3D detachment limited mode  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *simwe_mode*  
    **rain_intensity** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Rainfall intensity in mm/hr  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *50*  
    **rain_duration** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Total duration of storm event in minutes  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *60*  
    **precipitation** : str | io.StringIO, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Precipitation file  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input precipitation file  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, file, *name*  
    **k_factor** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;K factor  
    &nbsp;&nbsp;&nbsp;&nbsp;Soil erodibility factor  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **k_factor_value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;K factor constant  
    &nbsp;&nbsp;&nbsp;&nbsp;Soil erodibility constant  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.25*  
    **c_factor** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;C factor  
    &nbsp;&nbsp;&nbsp;&nbsp;Land cover factor  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **c_factor_value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;C factor constant  
    &nbsp;&nbsp;&nbsp;&nbsp;Land cover constant  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.1*  
    **m** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Water flow exponent  
    &nbsp;&nbsp;&nbsp;&nbsp;Water flow exponent  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.5*  
    **n** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Slope exponent  
    &nbsp;&nbsp;&nbsp;&nbsp;Slope exponent  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.2*  
    **walkers** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of walkers (max = 7000000)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1000000*  
    **runoff** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Runoff coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Runoff coefficient (0.6 for bare earth, 0.35 for grass or crops, 0.5 for shrubs and trees, 0.25 for forest, 0.95 for roads)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **runoff_value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Runoff coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Runoff coefficient (0.6 for bare earth, 0.35 for grass or crops, 0.5 for shrubs and trees, 0.25 for forest, 0.95 for roads)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.35*  
    **mannings** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Manning's roughness coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Manning's roughness coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **mannings_value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Manning's roughness coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Manning's roughness coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.04*  
    **detachment** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Detachment coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Detachment coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **detachment_value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Detachment coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Detachment coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.01*  
    **transport** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Transport coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Transport coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **transport_value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Transport coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Transport coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.01*  
    **shearstress** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Shear stress coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Shear stress coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **shearstress_value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Shear stress coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Shear stress coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.0*  
    **density** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Sediment mass density  
    &nbsp;&nbsp;&nbsp;&nbsp;Sediment mass density in g/cm^3  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **density_value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Sediment mass density  
    &nbsp;&nbsp;&nbsp;&nbsp;Sediment mass density in g/cm^3  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.4*  
    **mass** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Mass of sediment per unit area  
    &nbsp;&nbsp;&nbsp;&nbsp;Mass of sediment per unit area in kg/m^2  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **mass_value** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Mass of sediment per unit area  
    &nbsp;&nbsp;&nbsp;&nbsp;Mass of sediment per unit area in kg/m^2  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *116.*  
    **grav_diffusion** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Gravitational diffusion coefficient  
    &nbsp;&nbsp;&nbsp;&nbsp;Gravitational diffusion coefficient in m^2/s  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.1*  
    **erdepmin** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum values for erosion-deposition  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum values for erosion-deposition in kg/m^2s  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *-0.5*  
    **erdepmax** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum values for erosion-deposition  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum values for erosion-deposition in kg/m^2s  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0.5*  
    **start** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Start time in year-month-day hour:minute:second format  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2016-01-01 00:00:00*  
    **rain_interval** : int, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Time interval between evolution events in minutes  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **temporaltype** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;The temporal type of the space time dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *absolute, relative*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *absolute*  
    **threads** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of threads for multiprocessing  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **elevation_timeseries** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, strds, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *elevation_timeseries*  
    **depth_timeseries** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, strds, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *depth_timeseries*  
    **erdep_timeseries** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, strds, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *erdep_timeseries*  
    **flux_timeseries** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, strds, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *flux_timeseries*  
    **difference_timeseries** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output space time raster dataset  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, strds, *name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *difference_timeseries*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *f*  
    &nbsp;&nbsp;&nbsp;&nbsp;**f**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Fill depressions  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.sim.terrain* is a short-term landscape evolution model that simulates
topographic change for both steady state and dynamic flow regimes across
a range of spatial scales. It uses empirical models (RUSLE3D & USPED)
for soil erosion at watershed to regional scales and a physics-based
model (SIMWE) for shallow overland water flow and soil erosion at
subwatershed scales to compute short-term topographic change. This
either steady state or dynamic model simulates how overland sediment
mass flows reshape topography for a range of hydrologic soil erosion
regimes based on topographic, land cover, soil, and rainfall parameters.

## EXAMPLES

### Basic instructions

A basic example for the [North Carolina sample
dataset](https://grass.osgeo.org/download/data/). Install the add-on
module *r.sim.terrain*. Copy the raster elevation map *elev\_lid792\_1m*
from the PERMANENT mapset to the current mapset. Set the region to this
elevation map at 1 meter resolution. Run *r.sim.terrain* with the RUSLE
model for a 120 min event with a rainfall intensity of 50 mm/hr at a 3
minute interval. Set the empirical coefficients m and n to 0.4 and 1.3
respectively. Use the \`-f\` flag to fill depressions in order to reduce
the effect of positive feedback loops.

```sh
g.extension  extension=r.sim.terrain
g.copy raster=elev_lid792_1m@PERMANENT,elevation
g.region raster=elev_lid792_1m res=1
r.sim.terrain -f elevation=elevation runs=event mode=rusle_mode rain_intensity=50.0 rain_duration=120 rain_interval=3 m=0.4 n=1.3
```

![Net difference (m) for a dynamic RUSLE simulation](r_sim_terrain_rusle.png)  
*Figure: Net difference (m) for a dynamic RUSLE simulation of a 120 min
event with a rainfall intensity of 50 mm/hr with a 3 minute interval.*

### Spatially variable soil and landcover

Clone or download the [landscape evolution sample
dataset](https://github.com/baharmon/landscape_evolution_dataset) with a
time series of lidar-based digital elevation models and orthoimagery for
a highly eroded subwatershed of Patterson Branch Creek, Fort Bragg, NC,
USA.

Run *r.sim.terrain* with the simwe model for a 120 min event with a
rainfall intensity of 50 mm/hr. Use a transport value lower than the
detachment value to trigger a transport limited erosion regime. Use the
-f flag to fill depressions in order to reduce the effect of positive
feedback loops.

```sh
g.mapset -c mapset=transport location=nc_spm_evolution
g.region region=region res=1
r.mask vector=watershed
g.copy raster=elevation_2016@PERMANENT,elevation_2016
r.sim.terrain -f elevation=elevation_2016 runs=event mode=simwe_mode \
rain_intensity=50.0 rain_interval=120 rain_duration=10 walkers=1000000 \
detachment_value=0.01 transport_value=0.0001 manning=mannings runoff=runoff
```

![Net difference (m) for a steady state, transport limited SIMWE
simulation](r_sim_terrain.png)  
*Figure: Net difference (m) for a steady state, transport limited SIMWE
simulation of a 120 min event with a rainfall intensity of 50 mm/hr.*

For more detailed instructions and examples see this in-depth
[tutorial](https://github.com/baharmon/landscape_evolution/blob/master/tutorial.md).

## ERROR MESSAGES

If the module fails with

```text
ERROR: Unable to insert dataset of type raster in the temporal database. The mapset of the dataset does not match the current mapset.
```

check that the input **elevation** map is in the current mapset. The
input **elevation** map must be in the current mapset so that it can be
registered in the temporal database.

## REFERENCES

- Harmon, B. A., Mitasova, H., Petrasova, A., and Petras, V.:
    r.sim.terrain 1.0: a landscape evolution model with dynamic
    hydrology, Geosci. Model Dev., 12, 2837–2854,
    <https://doi.org/10.5194/gmd-12-2837-2019>, 2019.
- Mitasova H., Barton M., Ullah I., Hofierka J., Harmon R.S., 2013.
    [3.9 GIS-Based Soil Erosion
    Modeling](https://www.sciencedirect.com/science/article/abs/pii/B978012374739600052X).
    In J. F. Shroder, ed. Treatise on Geomorphology. San Diego: Academic
    Press, pp. 228-258. DOI:
    <https://doi.org/10.1016/B978-0-12-374739-6.00052-X>.

## SEE ALSO

*[r.sim.water](https://grass.osgeo.org/grass-stable/manuals/r.sim.water.html),
[r.sim.sediment](https://grass.osgeo.org/grass-stable/manuals/r.sim.sediment.html)*

## AUTHOR

Brendan A. Harmon  
Louisiana State University  
*<brendan.harmon@gmail.com>*

## SOURCE CODE

Available at: [r.sim.terrain source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/raster/r.sim.terrain)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/raster/r.sim.terrain))  
Latest change: Thursday Mar 20 21:36:57 2025 in commit [7286ecf](https://github.com/OSGeo/grass-addons/commit/7286ecf7af235bfd089fb9b1b82fb383cf95f3fc)
