---
name: r.suitability.regions.py
description: From suitability map to suitable regions
keywords: [  ]
---

# r.suitability.regions.py

From suitability map to suitable regions

=== "Command line"

    **r.suitability.regions.py**
    [**-dczakfvm**]
    **input**=*name*
    **output**=*name*
    [**suitability_threshold**=*float*]
    [**percentile_threshold**=*percentile*]
    [**percentage**=*percentage*]
    **minimum_size**=*float*
    [**minimum_suitability**=*float*]
    [**size**=*integer*]
    [**focal_statistic**=*string*]
    [**maximum_gap**=*float*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.suitability.regions.py input=name output=name suitability_threshold=float minimum_size=float
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.suitability.regions.py***",
        **input**,
        **output**,
        **suitability_threshold**=*None*,
        **percentile_threshold**=*None*,
        **percentage**=*None*,
        **minimum_size**,
        **minimum_suitability**=*None*,
        **size**=*1*,
        **focal_statistic**=*"median"*,
        **maximum_gap**=*"0"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.suitability.regions.py", input="name", output="name", suitability_threshold="float", minimum_size="float")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_suitability_regions_py*(**input**,
        **output**,
        **suitability_threshold**=*None*,
        **percentile_threshold**=*None*,
        **percentage**=*None*,
        **minimum_size**,
        **minimum_suitability**=*None*,
        **size**=*1*,
        **focal_statistic**=*"median"*,
        **maximum_gap**=*"0"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_suitability_regions_py(input="name", output="name", suitability_threshold="float", minimum_size="float")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Suitability raster  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster layer represting suitability (0-1)  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Output raster  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster with candidate regions for conservation  
    **suitability_threshold**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Threshold suitability score  
    &nbsp;&nbsp;&nbsp;&nbsp;The minimum suitability score to be included. For example, with a threshold of 0.7, all raster cells with a suitability of 0.7 are used as input in the delineation of contiguous suitable regions.  
    **percentile_threshold**=*percentile*  
    &nbsp;&nbsp;&nbsp;&nbsp;Percentile threshold (deprecated)  
    &nbsp;&nbsp;&nbsp;&nbsp;Percentile above which suitability scores are included in the search for suitable regions. For example, using a 0.95 percentile means that the raster cells with the 5% highest suitability scores are are used as input in the delineation of contiguous suitable regions (deprecated, use percentage instead).  
    **percentage**=*percentage*  
    &nbsp;&nbsp;&nbsp;&nbsp;Percentage to select  
    &nbsp;&nbsp;&nbsp;&nbsp;The percentage of raster cells with the highest suitability scores that are to be used as input in the delineation of contiguous suitable regions.  
    **minimum_size**=*float* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum area (in hectares)  
    &nbsp;&nbsp;&nbsp;&nbsp;Contiguous regions need to have a minimum area to be included.  
    **minimum_suitability**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Threshold for unsuitable areas  
    &nbsp;&nbsp;&nbsp;&nbsp;This option can be used to mark cells with a suitability equal or less than the given threshold as unsuitable. Can be used in conjuction with the 'focal statistics' option to ensure that those cells are marked as unsuitable (barriers), irrespective of the suitability scores of the surrounding cells.  
    **size**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Neighborhood size  
    &nbsp;&nbsp;&nbsp;&nbsp;The neighborhood size specifies which cells surrounding any given cell fall into the neighborhood for that cell. The size must be an odd integer and represent the length of one of moving window edges in cells. See the manual page of r.neighbors for more details  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **focal_statistic**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Neighborhood operation (focal statistic)  
    &nbsp;&nbsp;&nbsp;&nbsp;The median, maximum, first or 3rd quartile of the cells in a neighborhood of user-defined size is computed. This aggregated suitability score is used instead of the original suitability score to determine which raster cells are used as input in the delineation of contiguous suitable regions.  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *maximum, median, quart1, quart3*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *median*  
    **maximum_gap**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum gap size  
    &nbsp;&nbsp;&nbsp;&nbsp;Unsuitable areas (gaps) within suitable regions are removed if they are equal or smaller than the maximum size. This is done by merging them with the suitable regions in which they are located.  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **-d**  
    &nbsp;&nbsp;&nbsp;&nbsp;Clumps including diagonal neighbors  
    &nbsp;&nbsp;&nbsp;&nbsp;Diagonal neighboring cells are considerd to be connected, and will therefore be consiered part of the same region.  
    **-c**  
    &nbsp;&nbsp;&nbsp;&nbsp;Circular neighborhood for focal statistics  
    &nbsp;&nbsp;&nbsp;&nbsp;Use circular neighborhood when computing the focal statistic  
    **-z**  
    &nbsp;&nbsp;&nbsp;&nbsp;Average suitability per region  
    &nbsp;&nbsp;&nbsp;&nbsp;Create a map in which each region has a value corresponding to the average suitability of that region.  
    **-a**  
    &nbsp;&nbsp;&nbsp;&nbsp;Area of the regions  
    &nbsp;&nbsp;&nbsp;&nbsp;Create a map in which each region has a value corresponding to the surface area (hectares) of that region.  
    **-k**  
    &nbsp;&nbsp;&nbsp;&nbsp;Suitable areas  
    &nbsp;&nbsp;&nbsp;&nbsp;Map showing all raster cells with a suitability equal or above the user-defined threshold  
    **-f**  
    &nbsp;&nbsp;&nbsp;&nbsp;Suitable areas (focal statistics)  
    &nbsp;&nbsp;&nbsp;&nbsp;Map showing all raster cells with an aggregated suitability score based on a user-defined neighhborhood size that is equal or above a user-defined threshold.  
    **-v**  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector output layer  
    &nbsp;&nbsp;&nbsp;&nbsp;Create vector layer with suitabilty and compactness statistics  
    **-m**  
    &nbsp;&nbsp;&nbsp;&nbsp;Compute compactness of selected suitable regions.  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Suitability raster  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster layer represting suitability (0-1)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output raster  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster with candidate regions for conservation  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **suitability_threshold** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Threshold suitability score  
    &nbsp;&nbsp;&nbsp;&nbsp;The minimum suitability score to be included. For example, with a threshold of 0.7, all raster cells with a suitability of 0.7 are used as input in the delineation of contiguous suitable regions.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *float*  
    **percentile_threshold** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Percentile threshold (deprecated)  
    &nbsp;&nbsp;&nbsp;&nbsp;Percentile above which suitability scores are included in the search for suitable regions. For example, using a 0.95 percentile means that the raster cells with the 5% highest suitability scores are are used as input in the delineation of contiguous suitable regions (deprecated, use percentage instead).  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *percentile*  
    **percentage** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Percentage to select  
    &nbsp;&nbsp;&nbsp;&nbsp;The percentage of raster cells with the highest suitability scores that are to be used as input in the delineation of contiguous suitable regions.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *percentage*  
    **minimum_size** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum area (in hectares)  
    &nbsp;&nbsp;&nbsp;&nbsp;Contiguous regions need to have a minimum area to be included.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *float*  
    **minimum_suitability** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Threshold for unsuitable areas  
    &nbsp;&nbsp;&nbsp;&nbsp;This option can be used to mark cells with a suitability equal or less than the given threshold as unsuitable. Can be used in conjuction with the 'focal statistics' option to ensure that those cells are marked as unsuitable (barriers), irrespective of the suitability scores of the surrounding cells.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *float*  
    **size** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Neighborhood size  
    &nbsp;&nbsp;&nbsp;&nbsp;The neighborhood size specifies which cells surrounding any given cell fall into the neighborhood for that cell. The size must be an odd integer and represent the length of one of moving window edges in cells. See the manual page of r.neighbors for more details  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **focal_statistic** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Neighborhood operation (focal statistic)  
    &nbsp;&nbsp;&nbsp;&nbsp;The median, maximum, first or 3rd quartile of the cells in a neighborhood of user-defined size is computed. This aggregated suitability score is used instead of the original suitability score to determine which raster cells are used as input in the delineation of contiguous suitable regions.  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *maximum, median, quart1, quart3*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *median*  
    **maximum_gap** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum gap size  
    &nbsp;&nbsp;&nbsp;&nbsp;Unsuitable areas (gaps) within suitable regions are removed if they are equal or smaller than the maximum size. This is done by merging them with the suitable regions in which they are located.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *d*, *c*, *z*, *a*, *k*, *f*, *v*, *m*  
    &nbsp;&nbsp;&nbsp;&nbsp;**d**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Clumps including diagonal neighbors  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Diagonal neighboring cells are considerd to be connected, and will therefore be consiered part of the same region.  
    &nbsp;&nbsp;&nbsp;&nbsp;**c**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Circular neighborhood for focal statistics  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use circular neighborhood when computing the focal statistic  
    &nbsp;&nbsp;&nbsp;&nbsp;**z**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Average suitability per region  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Create a map in which each region has a value corresponding to the average suitability of that region.  
    &nbsp;&nbsp;&nbsp;&nbsp;**a**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Area of the regions  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Create a map in which each region has a value corresponding to the surface area (hectares) of that region.  
    &nbsp;&nbsp;&nbsp;&nbsp;**k**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Suitable areas  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Map showing all raster cells with a suitability equal or above the user-defined threshold  
    &nbsp;&nbsp;&nbsp;&nbsp;**f**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Suitable areas (focal statistics)  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Map showing all raster cells with an aggregated suitability score based on a user-defined neighhborhood size that is equal or above a user-defined threshold.  
    &nbsp;&nbsp;&nbsp;&nbsp;**v**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Vector output layer  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Create vector layer with suitabilty and compactness statistics  
    &nbsp;&nbsp;&nbsp;&nbsp;**m**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Compute compactness of selected suitable regions.  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Suitability raster  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster layer represting suitability (0-1)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output raster  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster with candidate regions for conservation  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **suitability_threshold** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Threshold suitability score  
    &nbsp;&nbsp;&nbsp;&nbsp;The minimum suitability score to be included. For example, with a threshold of 0.7, all raster cells with a suitability of 0.7 are used as input in the delineation of contiguous suitable regions.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *float*  
    **percentile_threshold** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Percentile threshold (deprecated)  
    &nbsp;&nbsp;&nbsp;&nbsp;Percentile above which suitability scores are included in the search for suitable regions. For example, using a 0.95 percentile means that the raster cells with the 5% highest suitability scores are are used as input in the delineation of contiguous suitable regions (deprecated, use percentage instead).  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *percentile*  
    **percentage** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Percentage to select  
    &nbsp;&nbsp;&nbsp;&nbsp;The percentage of raster cells with the highest suitability scores that are to be used as input in the delineation of contiguous suitable regions.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *percentage*  
    **minimum_size** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum area (in hectares)  
    &nbsp;&nbsp;&nbsp;&nbsp;Contiguous regions need to have a minimum area to be included.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *float*  
    **minimum_suitability** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Threshold for unsuitable areas  
    &nbsp;&nbsp;&nbsp;&nbsp;This option can be used to mark cells with a suitability equal or less than the given threshold as unsuitable. Can be used in conjuction with the 'focal statistics' option to ensure that those cells are marked as unsuitable (barriers), irrespective of the suitability scores of the surrounding cells.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *float*  
    **size** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Neighborhood size  
    &nbsp;&nbsp;&nbsp;&nbsp;The neighborhood size specifies which cells surrounding any given cell fall into the neighborhood for that cell. The size must be an odd integer and represent the length of one of moving window edges in cells. See the manual page of r.neighbors for more details  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **focal_statistic** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Neighborhood operation (focal statistic)  
    &nbsp;&nbsp;&nbsp;&nbsp;The median, maximum, first or 3rd quartile of the cells in a neighborhood of user-defined size is computed. This aggregated suitability score is used instead of the original suitability score to determine which raster cells are used as input in the delineation of contiguous suitable regions.  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *maximum, median, quart1, quart3*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *median*  
    **maximum_gap** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum gap size  
    &nbsp;&nbsp;&nbsp;&nbsp;Unsuitable areas (gaps) within suitable regions are removed if they are equal or smaller than the maximum size. This is done by merging them with the suitable regions in which they are located.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *0*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *d*, *c*, *z*, *a*, *k*, *f*, *v*, *m*  
    &nbsp;&nbsp;&nbsp;&nbsp;**d**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Clumps including diagonal neighbors  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Diagonal neighboring cells are considerd to be connected, and will therefore be consiered part of the same region.  
    &nbsp;&nbsp;&nbsp;&nbsp;**c**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Circular neighborhood for focal statistics  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use circular neighborhood when computing the focal statistic  
    &nbsp;&nbsp;&nbsp;&nbsp;**z**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Average suitability per region  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Create a map in which each region has a value corresponding to the average suitability of that region.  
    &nbsp;&nbsp;&nbsp;&nbsp;**a**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Area of the regions  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Create a map in which each region has a value corresponding to the surface area (hectares) of that region.  
    &nbsp;&nbsp;&nbsp;&nbsp;**k**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Suitable areas  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Map showing all raster cells with a suitability equal or above the user-defined threshold  
    &nbsp;&nbsp;&nbsp;&nbsp;**f**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Suitable areas (focal statistics)  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Map showing all raster cells with an aggregated suitability score based on a user-defined neighhborhood size that is equal or above a user-defined threshold.  
    &nbsp;&nbsp;&nbsp;&nbsp;**v**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Vector output layer  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Create vector layer with suitabilty and compactness statistics  
    &nbsp;&nbsp;&nbsp;&nbsp;**m**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Compute compactness of selected suitable regions.  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

Multi-criteria or suitability analyses are useful methods to map the
relative suitability. For example, they can be used to map the relative
habitat suitability of a species, based on multiple criteria. A typical
outcome of such analyses is a raster layer with suitability scores
between 0 (not suitable) and 1 (very suitable).

Often, the next step is to use this suitability map to identify suitable
area/region, e.g., to delineate potential areas for nature conservation.
With this addon you can identify regions of contiguous cells that have a
suitability score above a certain threshold and a minimum size. There
are a number of additional options explored below.

### Option 1 - basic use

The user defines a threshold suitability score. All raster cells with a
suitability score equal or above the threshold are reclassified as
suitable. All other raster cells are reclassified to NODATA. Next, all
contiguous raster cells (i.e., all neighboring rastercells) are clumped.
Clumps below an user-defined size (minimum area for fragments) are
subsequently removed. See
[r.reclass.area](https://grass.osgeo.org/grass-stable/manuals/r.reclass.area.html)
for more details.

![image-alt](r_suitability_regions_01.png)*  
Figure 1: identifying regions of contiguous raster cells with a
suitability score of 0.7 or more.*

You would use this to find suitable areas for a species that cannot or
is not likely to venture into areas where conditions are not optimal.

### Option 2 - focal area

To consider the requirements (of e.g., a species) at a landscape scale,
the habitat suitability of the surrounding cells can be taken into
account as well. This is done by first computing a raster where the
value for each output cell is a function (maximum, median, 1st quartile
or 3rd quartile) of the values of all the input cells in a user-defined
neighborhood. For example, take the 5x5 neighborhood below. Using the
maximum as statistic, the central cell would be assigned a value of 5.
Using the median, it would be assigned the value 2. See
[r.neighbors](https://grass.osgeo.org/grass-stable/manuals/r.neighbors.html)
for more details.

```text
1 1 2 2 1
4 4 1 3 1
1 3 2 1 4
5 2 1 3 2
1 2 3 2 1
```

Next, the resulting output map is used instead of the original
suitability map to identify the raster cells with a score equal to or
above the user-defined threshold value. So, raster cells are selected if
they have a suitability score equal or above the threshold value, or if
at least one raster cell (maximum), half of the raster cells (median),
25% of the raster cells (1st quartile), or 75% of the raster cells (3rd
quartile) in the neighborhood have a suitability score equal or above
the given threshold.

As in the first use case, the selected raster cells are clumped into
contiguous regions, and regions that are smaller than an user-defined
size are removed. This option would be a good choice if the target
species has no problem to briefly stay in non-suitable habitat, e.g., to
cross it on their way to more suitable habitat. As the example below
shows, it results in larger regions than in the previous option.

![image-alt](r_suitability_regions_02.png)*  
Figure 2: Like figure 1, but based on the median suitability scores of
the neighboring cells within a radius of 300 meter (3x3 moving window).*

### Option 3 - barriers

The user has the option to define an absolute minimum suitability score.
Raster cells with a suitability below this score are always considered
unsuitable, irrespective of the suitability scores of the surrounding
raster cells. This option only affects the results of option 2.

The minimum suitability score can be used to identify barriers or areas
where a species cannot cross. For example, a road can break up larger
regions of otherwise suitable habitats into smaller fragments. For
species that cannot cross roads, this effectively results in smaller
isolated populations rather than one large (meta-)population. It can
even result in a net loss of habitat if one or more of the fragments are
too small to maintain a population (the user can set a minimum area size
to account for this).

![image-alt](r_suitability_regions_03.png)*  
Figure 3: Like figure 2, but considering raster cells with suitability 0
(mostly roads) as absolute barriers. Diagonally connected raster cells
are not considered to form a contiguous region.*

Note that for line elements like roads, results may differ if the option
to 'include the diagonal neighbors when defining clumps' (flag d) is
selected. For example, in figure 4, diagonally connected cells are
considered as neighbors. As a consequence, the suitable areas on both
sides of the road are considered to be part of the same region. I.e.,
the road does not act as a barrier here.

![image-alt](r_suitability_regions_04.png)*  
Figure 4: Like figure 3, but this time, diagonally connected raster
cells are considered to form a contiguous region.*

### Option 4 - compact areas

The user can opt to include patches of unsuitable areas that fall within
suitable regions into the final selection of suitable regions. Only gaps
smaller than a user-defined maximum size will be included.

This option can be used to end up with more compact areas. This may be
desirable for visualisation purposes, or it may in fact be acceptable to
include such areas in the final selection of a region.

![image-alt](r_suitability_regions_05.png)*  
Figure 5: Like figure 3 (left), but here gaps (areas within a suitable
region) of 500 hectares or less were included in the final selection
(middle). The right map shows the suitable areas within the selected
regions (green) and the filled gaps (yellow).*

Selecting this option will generate a second map which shows the 'filled
patches'. This makes it easier to e.g., inspect the feasibility or
desirability to actually include these areas in a protected area.

### Compactness of the regions

To compare the compactness of the resulting regions, the compactness of
an area is calculated using the formula below (see also
[v.to.db](https://grass.osgeo.org/grass-stable/manuals/v.to.db.html).

```text
compactness = perimeter / (2 * sqrt(PI * area))
```

This will create a layer with the basename with the suffix
'compactness'. The compactness will also be calculated as one of the
region statistics if the option to save the result as a vector layer is
selected (see under 'other options' below.

### Other options

The user can opt to save two intermediate layers: the layer showing all
raster cells with a suitability higher than the threshold (flag k; file
name with the suffix *\_allsuitableareas*), and the layer with the
suitability based on focal statistics (flag f; file name with suffix
*\_focalsuitability*). There is furthermore the option to create a layer
with the average suitability per clump (flag z), and a layer with the
surface area (in hectares) of the clumped regions (flag a).

Selecting the 'v' flag will create a vector layer with the regions. The
attribute table of this vector layer will include columns with the
surface area (m2), compactness, fractal dimension (*fd*), and average
suitability. For the meaning of compactness, see above. The fractal
dimension of the boundary of a polygon is calculated using the formula
below (see also
[v.to.db](https://grass.osgeo.org/grass-stable/manuals/v.to.db.html).

```text
fd = 2 * (log(perimeter) / log(area))
```

## NOTE

This addon uses the *r.reclass.area* function to find the clumps. Like
in that function, the user can opt to consider diagonally connected
raster cells to be part of a contiguous region. Using this option will
in most cases result in less compact regions. It may furthermore result
in regions that would otherwise be considered as separate regions to
appear as one large region instead.

The option to calculate the area of clumped regions should only be used
with projected layers because the assumption is that all cells have the
same size.

## Examples

See [this
tutorial](https://ecodiv.earth/TutorialsNotes/SuitabilityRegions/index.html)
for examples.

## See also

- [r.reclass.area](https://grass.osgeo.org/grass-stable/manuals/r.reclass.area.html)
- [r.clump](https://grass.osgeo.org/grass-stable/manuals/r.clump.html)
- [v.to.db](https://grass.osgeo.org/grass-stable/manuals/v.to.db.html)
- [r.area](r.area.md)

## REQUIREMENTS

If you use the option to compute the area per clump (using the **-a**
flag), you need to install the *r.area* module first.

## AUTHOR

Paulo van Breugel, paulo at ecodiv.earth  
[HAS green academy](https://has.nl)  
[Innovative biomonitoring research
group](https://www.has.nl/en/research/professorships/innovative-bio-monitoring-professorship/)  
[Climate-robust Landscapes research
group](https://www.has.nl/en/research/professorships/climate-robust-landscapes-professorship/)

## SOURCE CODE

Available at: [r.suitability.regions source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/raster/r.suitability.regions)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/raster/r.suitability.regions))  
Latest change: Thursday Mar 20 21:36:57 2025 in commit [7286ecf](https://github.com/OSGeo/grass-addons/commit/7286ecf7af235bfd089fb9b1b82fb383cf95f3fc)
