---
name: r.timeofconcentration.py
description: Computes per-cell time of concentration (Tc) using the Kirpich equation from longest upstream flow-path length and path-average slope.
keywords: [ raster, hydrology, time of concentration, SCS ]
---

# r.timeofconcentration.py

Computes per-cell time of concentration (Tc) using the Kirpich equation from longest upstream flow-path length and path-average slope.

=== "Command line"

    **r.timeofconcentration.py**
    **elevation**=*name*
    **direction**=*name*
    **streams**=*name*
    **time_concentration**=*name*
    [**outlets**=*name*]
    [**slope_min**=*float*]
    [**length_min**=*float*]
    [**vertical_units**=*string*]
    [**factor**=*float*]
    [**length**=*name*]
    [**drop**=*name*]
    [**sbar**=*name*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.timeofconcentration.py elevation=name direction=name streams=name time_concentration=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.timeofconcentration.py***",
        **elevation**,
        **direction**,
        **streams**,
        **time_concentration**,
        **outlets**=*None*,
        **slope_min**=*1e-4*,
        **length_min**=*10*,
        **vertical_units**=*"meters"*,
        **factor**=*None*,
        **length**=*None*,
        **drop**=*None*,
        **sbar**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.timeofconcentration.py", elevation="name", direction="name", streams="name", time_concentration="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_timeofconcentration_py*(**elevation**,
        **direction**,
        **streams**,
        **time_concentration**,
        **outlets**=*None*,
        **slope_min**=*1e-4*,
        **length_min**=*10*,
        **vertical_units**=*"meters"*,
        **factor**=*None*,
        **length**=*None*,
        **drop**=*None*,
        **sbar**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_timeofconcentration_py(elevation="name", direction="name", streams="name", time_concentration="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **elevation**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input elevation raster map [m]  
    **direction**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input flow direction raster map (from r.watershed or r.stream.extract)  
    **streams**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input stream raster map consistent with 'direction' (from r.watershed or r.stream.extract)  
    **time_concentration**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output time of concentration raster map [hours]  
    **outlets**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of optional input outlets raster map; when set, Tc is reported only at these cells  
    **slope_min**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum path-average slope (unitless) to avoid division by zero on flats  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1e-4*  
    **length_min**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum upstream flow-path length to report Tc [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *10*  
    **vertical_units**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Vertical units of elevation raster (converted to meters internally)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *meters, feet, factor*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *meters*  
    **factor**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Conversion factor to meters when vertical_units=factor (ensure factor \* units = meters)  
    **length**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for optional output longest upstream flow-path length raster map L [m]  
    **drop**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for optional output flow-path elevation drop raster map delta_z (&gt;= 0) [m]  
    **sbar**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for optional output path-average slope raster map S_bar = max(delta_z / L, slope_min) [unitless]  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **elevation** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input elevation raster map [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **direction** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input flow direction raster map (from r.watershed or r.stream.extract)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **streams** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input stream raster map consistent with 'direction' (from r.watershed or r.stream.extract)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **time_concentration** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output time of concentration raster map [hours]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **outlets** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of optional input outlets raster map; when set, Tc is reported only at these cells  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **slope_min** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum path-average slope (unitless) to avoid division by zero on flats  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1e-4*  
    **length_min** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum upstream flow-path length to report Tc [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *10*  
    **vertical_units** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Vertical units of elevation raster (converted to meters internally)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *meters, feet, factor*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *meters*  
    **factor** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Conversion factor to meters when vertical_units=factor (ensure factor \* units = meters)  
    **length** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for optional output longest upstream flow-path length raster map L [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **drop** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for optional output flow-path elevation drop raster map delta_z (&gt;= 0) [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **sbar** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for optional output path-average slope raster map S_bar = max(delta_z / L, slope_min) [unitless]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **elevation** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input elevation raster map [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **direction** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input flow direction raster map (from r.watershed or r.stream.extract)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **streams** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input stream raster map consistent with 'direction' (from r.watershed or r.stream.extract)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **time_concentration** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output time of concentration raster map [hours]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **outlets** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of optional input outlets raster map; when set, Tc is reported only at these cells  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **slope_min** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum path-average slope (unitless) to avoid division by zero on flats  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1e-4*  
    **length_min** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Minimum upstream flow-path length to report Tc [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *10*  
    **vertical_units** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Vertical units of elevation raster (converted to meters internally)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *meters, feet, factor*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *meters*  
    **factor** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Conversion factor to meters when vertical_units=factor (ensure factor \* units = meters)  
    **length** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for optional output longest upstream flow-path length raster map L [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **drop** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for optional output flow-path elevation drop raster map delta_z (&gt;= 0) [m]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **sbar** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for optional output path-average slope raster map S_bar = max(delta_z / L, slope_min) [unitless]  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.timeofconcentration* generates a raster map of time of
concentration (Tc) [h] for each cell, using the Kirpich equation based on the
longest upstream flow-path length and path-average slope. This method, rooted
in hydrologic practice, estimates how long it takes water to travel from the
farthest point in a watershed to a given cell, aiding in runoff and flood
analysis. The tool leverages elevation and flow direction data, optionally
deriving streams if not provided, and supports diagnostic outputs like flow
length and slope. All measurements use metric units ([m], [h]).

It simplifies water travel time estimation for high-level watershed planning:
water flows downhill along paths defined by terrain, and Tc reflects the
slowest route’s duration, influenced by distance and slope steepness. This is
key for understanding how quickly runoff reaches streams or outlets.

### Inputs

**elevation:** Raster of elevation [m]. Required input to derive terrain
slopes and flow paths.

**direction:** Flow-direction raster (GRASS-coded; from `r.watershed` or
`r.stream.extract`). Required to trace upstream flow paths; defines the
drainage network for accumulation.

**streams:** Stream raster and **must be consistent** with `direction` (i.e.,
produced from the same run of `r.watershed` or from `r.stream.extract` on the
same DEM).

**outlets:** Optional raster of outlet points; Tc is computed only at these
cells (NULL elsewhere), respecting the current mask/region.

**slope_min:** Minimum path-average slope [unitless] ($10^{-4}$ default).
Prevents division by zero on flat areas by setting a floor value.

**length_min:** Minimum upstream flow-path length [m] ($10$ default). Ensures
Tc is reported only where flow paths are significant.

**vertical_units:** optional parameter that defines the vertical units of the
elevation raster. It accepts three values: `meters`, `feet`, or `factor`, and
defaults to `meters`. When `vertical_units=meters`, no conversion is applied.
When `vertical_units=feet`, elevation drops are multiplied by `1/3.28084` to
convert them to meters. When `vertical_units=factor`, the module multiplies
elevation drops by the user-supplied factor.

**factor:** Required only when `vertical_units=factor`. It must convert the
user’s vertical unit to meters (`factor × <your vertical unit> = meters`), and
the user is responsible for providing the correct factor.

### Outputs

**output:** A raster map of time of concentration per cell [h], computed using
the Kirpich equation:

$$
T_c \;=\; \frac{K \cdot L^a \cdot S_{\text{avg}}^b}{60}
$$

where $L$ is upstream flow-path length [m], $S_{\text{avg}}$ is path-average
slope [unitless], $K = 0.01947$, $a = 0.77$, $b = -0.385$ are Kirpich
constants, and the result is converted from minutes to hours by dividing by 60.
Tc is NULL if L < length_min or outlets are undefined.

**length:** Optional output raster of longest upstream flow-path length per
cell [m], derived from `r.stream.distance`.

**drop:** Optional output raster of flow-path elevation drop per cell [m]
($\geq 0$), computed as the maximum elevation difference.

**sbar:** Optional raster of path-average slope per cell [unitless], calculated
as $S_{\text{avg}} = \max(\frac{\Delta z}{L}, \text{slope}_\text{{min}})$,
where $\Delta z$ is the drop and $L$ is the length.

## Notes

- **CRS-aware distance** `r.stream.distance` returns lengths in meters,
  even if the CRS is geographic or projected (meters or feet).
- **Static Tc** Tc is a steady-state estimate; no temporal
  routing or storage is modeled.

## EXAMPLE

These examples use the North Carolina sample dataset.

Calculate time of concentration using r.watershed and r.timeofconcentration:

```sh
# set the region
g.region -p raster=elevation

# calculate positive flow accumulation and drainage directions using r.watershed
r.watershed -sa elevation=elevation drainage=fdr stream=str threshold=10

# compute the time of concentration
r.timeofconcentration elevation=elevation direction=fdr streams=str tc=tc_nc

# use length_min parameter for coarser tc on important streams only
r.timeofconcentration elevation=elevation direction=fdr streams=str tc=tc_nc_250 length_min=250

# if the vertical units of the DEM are in feet
r.timeofconcentration elevation=elevation vunits=feet dir=fdr str=str tc=tc

# if the vertical units of the DEM are in units other than meters or feet (e.g., cm)
r.timeofconcentration elevation=elevation vunits=factor factor=0.01 dir=fdr str=str tc=tc
```

![r_timeofconcentration example](tc_nc.png)
*Figure: Output from r.timeofconcentration with length_min=250 on NC dataset
zoomed near the watershed outlet*

## REFERENCES

1. Kirpich, Z. P. (1940). Time of concentration of small agricultural
   watersheds. Civil Eng., 10(6), 362.
2. United States Department of Agriculture, Natural Resources Conservation
   Service. (2008). National Engineering Handbook, Part 630 Hydrology: Chapter
   15 – Time of Concentration (210-VI-NEH)

## SEE ALSO

[r.watershed](https://grass.osgeo.org/grass-stable/manuals/r.watershed.html),
[r.stream.distance](https://grass.osgeo.org/grass-stable/manuals/addons/r.stream.distance.html),
[r.curvenumber](https://grass.osgeo.org/grass-stable/manuals/addons/r.curvenumber.html),
[r.runoff](https://grass.osgeo.org/grass-stable/manuals/addons/r.runoff.html)

## AUTHORS

[Abdullah Azzam](mailto:mabdazzam@outlook.com)
([CLAWRIM](https://clawrim.isnew.info/), Department of Civil and Environmental
Engineering, New Mexico State University)

## SOURCE CODE

Available at: [r.timeofconcentration source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/raster/r.timeofconcentration)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/raster/r.timeofconcentration))  
Latest change: Monday Dec 15 10:14:24 2025 in commit [1a98df1](https://github.com/OSGeo/grass-addons/commit/1a98df109d4bcb6a89413b6f976b8da774da06a3)
