---
name: r.colors.matplotlib.py
description: Convert or apply a Matplotlib color table to a GRASS raster map
keywords: [ raster, color table, matplotlib ]
---

# r.colors.matplotlib.py

Convert or apply a Matplotlib color table to a GRASS raster map

=== "Command line"

    **r.colors.matplotlib.py**
    [**-dngae**]
    [**map**=*name* [,*name*,...]]
    [**output**=*name*]
    [**color**=*string*]
    [**ncolors**=*integer*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.colors.matplotlib.py
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.colors.matplotlib.py***",
        **map**=*None*,
        **output**=*None*,
        **color**=*None*,
        **ncolors**=*6*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.colors.matplotlib.py")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_colors_matplotlib_py*(**map**=*None*,
        **output**=*None*,
        **color**=*None*,
        **ncolors**=*6*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_colors_matplotlib_py()
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **map**=*name* [,*name*,...]  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map(s) to apply color table to  
    **output**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for the new color table rules file  
    **color**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of color table  
    &nbsp;&nbsp;&nbsp;&nbsp;Available color tables depend on the Matplotlib version. Alternatively this can be file name of a file generated by Python viscm tool  
    **ncolors**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of colors in the color table  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of color intervals in a discrete color table with -d  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *2-*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *6*  
    **-d**  
    &nbsp;&nbsp;&nbsp;&nbsp;Generate discrete color table  
    &nbsp;&nbsp;&nbsp;&nbsp;Generate discrete (interval) color table instead of a continuous one  
    **-n**  
    &nbsp;&nbsp;&nbsp;&nbsp;Reverse the order of colors (invert colors)  
    **-g**  
    &nbsp;&nbsp;&nbsp;&nbsp;Logarithmic scaling  
    **-a**  
    &nbsp;&nbsp;&nbsp;&nbsp;Logarithmic-absolute scaling  
    **-e**  
    &nbsp;&nbsp;&nbsp;&nbsp;Histogram equalization  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **map** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map(s) to apply color table to  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for the new color table rules file  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    **color** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of color table  
    &nbsp;&nbsp;&nbsp;&nbsp;Available color tables depend on the Matplotlib version. Alternatively this can be file name of a file generated by Python viscm tool  
    **ncolors** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of colors in the color table  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of color intervals in a discrete color table with -d  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *2-*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *6*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *d*, *n*, *g*, *a*, *e*  
    &nbsp;&nbsp;&nbsp;&nbsp;**d**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Generate discrete color table  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Generate discrete (interval) color table instead of a continuous one  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Reverse the order of colors (invert colors)  
    &nbsp;&nbsp;&nbsp;&nbsp;**g**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Logarithmic scaling  
    &nbsp;&nbsp;&nbsp;&nbsp;**a**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Logarithmic-absolute scaling  
    &nbsp;&nbsp;&nbsp;&nbsp;**e**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Histogram equalization  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **map** : str | list[str], *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster map(s) to apply color table to  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for the new color table rules file  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, file, *name*  
    **color** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of color table  
    &nbsp;&nbsp;&nbsp;&nbsp;Available color tables depend on the Matplotlib version. Alternatively this can be file name of a file generated by Python viscm tool  
    **ncolors** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of colors in the color table  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of color intervals in a discrete color table with -d  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *2-*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *6*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *d*, *n*, *g*, *a*, *e*  
    &nbsp;&nbsp;&nbsp;&nbsp;**d**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Generate discrete color table  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Generate discrete (interval) color table instead of a continuous one  
    &nbsp;&nbsp;&nbsp;&nbsp;**n**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Reverse the order of colors (invert colors)  
    &nbsp;&nbsp;&nbsp;&nbsp;**g**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Logarithmic scaling  
    &nbsp;&nbsp;&nbsp;&nbsp;**a**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Logarithmic-absolute scaling  
    &nbsp;&nbsp;&nbsp;&nbsp;**e**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Histogram equalization  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

The *r.colors.matplotlib* module converts
[Matplotlib](https://matplotlib.org/) color maps to GRASS color table
format (rules) and assigns it to a given raster map. The created color
table is always relative (color rules with percent) When option **map**
is specified *r.colors.matplotlib* assigns the color rules to the given
raster map. The color tables is always stretched based on the range of
values of the map

Depending on the use case, it may be advantageous to use the **-d** to
discretize the color table into intervals.

![image-alt](r_colors_matplotlib_gist_earth.png)
![image-alt](r_colors_matplotlib_gist_earth_discrete.png)

*Continuous (default) and discrete (**-d**) color table*

## NOTES

This module depends on *[Matplotlib](https://matplotlib.org/)* which
needs to be installed on your computer. Use your Python package manager
(e.g. *pip*) or distribution package manager to install it.

The selection of color tables depends on the Matplotlib version. Note
that the perceptually uniform sequential color tables, namely *viridis*,
*inferno*, *plasma*, and *magma*, are available in Matplotlib 1.5 and
above. Color tables are called *color maps* (or colormaps) in Matplotlib
and the best overview of available color maps in the
[colormaps\_reference](https://matplotlib.org/stable/users/explain/colors/colormaps.html)
example in Matplotlib documentation.

## EXAMPLES

### Creating a color table as GRASS color rules

Convert *summer* color table to GRASS color table rules format. If we
don't specify output file, it is printed to standard output. We set
number of colors to 2 because that's enough for this given color table
(it has one color at the beginning and one at the end and linear
interpolation can be used for the values in between).

```sh
r.colors.matplotlib color=summer ncolors=2
```

```text
0.000% 0:127:102
100.000% 255:255:102
```

In case we want to use a discrete color table with intervals with given
constant color, we use the **-d** flag and the number of colors is now
the number of intervals, so we want to make it higher, 5 in this case.

```sh
r.colors.matplotlib color=summer ncolors=5 -d
```

```text
0.000% 0:127:102
20.000% 0:127:102
20.000% 63:159:102
40.000% 63:159:102
40.000% 127:191:102
60.000% 127:191:102
60.000% 191:223:102
80.000% 191:223:102
80.000% 255:255:102
100.000% 255:255:102
```

### Setting color table for a raster map

Now we set several different color tables for the elevation raster map
from the North Carolina sample dataset. We use continuous and discrete
color tables (gradients). The color tables are stretched to fit the
raster map range.

```sh
r.colors.matplotlib color=summer map=elevation
r.colors.matplotlib color=winter ncolors=8 map=elevation -d
r.colors.matplotlib color=autumn map=elevation
r.colors.matplotlib color=cubehelix ncolors=8 map=elevation -d
r.colors.matplotlib color=terrain map=elevation
```

We can display legend:

```sh
d.legend raster=elevation labelnum=10 at=5,50,7,10
```

![image-alt](r_colors_matplotlib.png)
![image-alt](r_colors_matplotlib_winter_discrete.png)
![image-alt](r_colors_matplotlib_autumn.png)
![image-alt](r_colors_matplotlib_cubehelix_discrete.png)
![image-alt](r_colors_matplotlib_terrain.png)

*summer, winter, autumn, cubehelix, and terrain color tables applied to
the elevation raster from the North Carolina sample dataset. winter and
cubehelix are set to be discrete instead of continuous.*

### Setting color table for a vector map

First we create a text file with color rules:

```sh
r.colors.matplotlib color=summer output=mpl_summer.txt
```

Then we set color table for the vector to the rules stored in a file:

```sh
v.colors map=points rules=mpl_summer.txt
```

Color table for 3D raster map can be set in the same way.

### Using color tables generated by the viscm tool

A [viscm](https://pypi.org/project/viscm) tool is a little tool for
analyzing color tables and creating new color tables (color maps) for
Matplotlib. The tool was used to create perceptually uniform color
tables for Matplotlib (for example viridis). The new color table is
stored into a file. In version 0.7, a temporary file named
`/tmp/new_cm.py` which is a Python source code which creates a
*Colormap* object. If this module gets a name of existing file instead
of a color table name, it assumes that it this kind of file and reads
object called `test_cm` as Matplotlib color table. The possible workflow
follows. (Note that you need to install the viscm tool, e.g. using `sudo
pip install viscm` on Linux.)

Start the tool, create and save a color table:

```sh
python -m viscm edit
```

Now store the color table in GRASS GIS format:

```sh
r.colors.matplotlib color=/tmp/new_cm.py rules=from_viscm.txt
```

![image-alt](r_colors_matplotlib_viscm_edit.png)
![image-alt](r_colors_matplotlib_viscm_properties.png)

*Editing color table in viscm (right): the yellow dot on the blue spline
must stay in the colored area as the red line moves. Reviewing color
table properties is done using several displays including color
blindness simulations.*

![image-alt](r_colors_matplotlib_viscm.png)

*A color table from viscm applied to the elevation raster from the North
Carolina sample dataset.*

The same works for any Python files which follow the same schema, so it
works for example with files from the
[BIDS/colormap](https://github.com/BIDS/colormap) repository.

## SEE ALSO

*[r.colors](https://grass.osgeo.org/grass-stable/manuals/r.colors.html),
[v.colors](https://grass.osgeo.org/grass-stable/manuals/v.colors.html),
[r3.colors](https://grass.osgeo.org/grass-stable/manuals/r3.colors.html),
[r.cpt2grass](r.cpt2grass.md),
[r.colors.cubehelix](r.colors.cubehelix.md)*

[colormaps\_reference](https://matplotlib.org/stable/users/explain/colors/colormaps.html)
example in Matplotlib documentation

## AUTHOR

Vaclav Petras, [NCSU GeoForAll
Lab](https://geospatial.ncsu.edu/geoforall/)

## SOURCE CODE

Available at: [r.colors.matplotlib source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/raster/r.colors.matplotlib)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/raster/r.colors.matplotlib))  
Latest change: Thursday Mar 20 21:36:57 2025 in commit [7286ecf](https://github.com/OSGeo/grass-addons/commit/7286ecf7af235bfd089fb9b1b82fb383cf95f3fc)
