---
name: r.divergence.py
description: Computes divergence of a vector field defined by magnitude and direction
keywords: [ raster, divergence ]
---

# r.divergence.py

Computes divergence of a vector field defined by magnitude and direction

=== "Command line"

    **r.divergence.py**
    **magnitude**=*name*
    **direction**=*name*
    **output**=*name*
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.divergence.py magnitude=name direction=name output=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.divergence.py***",
        **magnitude**,
        **direction**,
        **output**,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.divergence.py", magnitude="name", direction="name", output="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_divergence_py*(**magnitude**,
        **direction**,
        **output**,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_divergence_py(magnitude="name", direction="name", output="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **magnitude**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map representing magnitude  
    **direction**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map representing direction  
    &nbsp;&nbsp;&nbsp;&nbsp;Direction is in degrees ccw from the east  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of output divergence raster  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **magnitude** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map representing magnitude  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **direction** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map representing direction  
    &nbsp;&nbsp;&nbsp;&nbsp;Direction is in degrees ccw from the east  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of output divergence raster  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **magnitude** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map representing magnitude  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **direction** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input raster map representing direction  
    &nbsp;&nbsp;&nbsp;&nbsp;Direction is in degrees ccw from the east  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of output divergence raster  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

Module *r.divergence* computes the divergence of a vector field given by
**magnitude** and **direction** raster maps. Direction is in degrees
counterclockwise from the east and can be computed using
*[r.slope.aspect](https://grass.osgeo.org/grass-stable/manuals/r.slope.aspect.html)*.
This module can be used for estimating erosion and deposition rates for
a steady state overland flow using USPED (Unit Stream Power based
Erosion Deposition) model. Net erosion/deposition is estimated as a
change in sediment flow rate expressed by a divergence in sediment flow.

## EXAMPLES

In North Carolina sample dataset, we compute net erosion/deposition.

```sh
g.region raster=elev_lid792_1m -p
r.slope.aspect elevation=elev_lid792_1m slope=slope aspect=aspect
r.flow elevation=elev_lid792_1m flowaccumulation=flowacc

# exponents m=1.3 and n=1.2
# multiply flowaccumulation by cell area/resolution to get contributing area per unit width
r.mapcalc "sflowtopo = pow(flowacc * 1.,1.3) * pow(sin(slope),1.2)"

# Compute sediment flow by combining the rainfall, soil and land cover factors
# with the topographic sediment transport factor.
# We use a constant value of 270. for rainfall intensity factor.
r.mapcalc "sedflow = 270. * soils_Kfactor * cfactorbare_1m * sflowtopo"
r.divergence magnitude=sedflow direction=aspect output=erosion_deposition

# set suitable color table
r.colors map=erosion_deposition rules=- << EOF
0% 100 0 100   #dark magenta
-100 magenta
-10 red
-1 orange
-0.1 yellow
0 200 255 200     #light green
0.1 cyan
1 aqua
10 blue
100 0 0 100       #dark blue
100% black
EOF
```

## SEE ALSO

*[r.slope.aspect](https://grass.osgeo.org/grass-stable/manuals/r.slope.aspect.html)*

## AUTHORS

Anna Petrasova, [NCSU GeoForAll
Lab](https://geospatial.ncsu.edu/geoforall/)  
Helena Mitasova, [NCSU GeoForAll
Lab](https://geospatial.ncsu.edu/geoforall/)

## SOURCE CODE

Available at: [r.divergence source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/raster/r.divergence)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/raster/r.divergence))  
Latest change: Thursday Feb 20 13:02:26 2025 in commit [53de819](https://github.com/OSGeo/grass-addons/commit/53de8196a10ba5a8a9121898ce87861d227137e3)
