---
name: r.hand.py
description: Performs Height Above Nearest Drainage (HAND) analysis and flood inundation mapping with HAND method.
keywords: [ raster, hydrology, flood, inundation, json ]
---

# r.hand.py

Performs Height Above Nearest Drainage (HAND) analysis and flood inundation mapping with HAND method.

=== "Command line"

    **r.hand.py**
    [**-tm**]
    **elevation**=*name*
    [**streams**=*name*]
    [**direction**=*name*]
    [**inundation_raster**=*name*]
    [**hand**=*name*]
    [**inundation_strds**=*name*]
    [**threshold**=*integer*]
    [**depth**=*float*]
    [**start_water_level**=*float*]
    [**end_water_level**=*float*]
    [**water_level_step**=*float*]
    [**memory**=*memory in MB*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.hand.py elevation=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.hand.py***",
        **elevation**,
        **streams**=*None*,
        **direction**=*None*,
        **inundation_raster**=*None*,
        **hand**=*None*,
        **inundation_strds**=*None*,
        **threshold**=*10000*,
        **depth**=*None*,
        **start_water_level**=*None*,
        **end_water_level**=*None*,
        **water_level_step**=*1*,
        **memory**=*300*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.hand.py", elevation="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_hand_py*(**elevation**,
        **streams**=*None*,
        **direction**=*None*,
        **inundation_raster**=*None*,
        **hand**=*None*,
        **inundation_strds**=*None*,
        **threshold**=*10000*,
        **depth**=*None*,
        **start_water_level**=*None*,
        **end_water_level**=*None*,
        **water_level_step**=*1*,
        **memory**=*300*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_hand_py(elevation="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **elevation**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input elevation raster map  
    **streams**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Stream raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the stream raster map  
    **direction**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Flow direction raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the flow direction raster map  
    **inundation_raster**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output inundation raster map  
    **hand**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Height above nearest drainage raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output HAND raster map  
    **inundation_strds**=*name*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output inundation STRDS.  
    **threshold**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basin threshold value  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *10000*  
    **depth**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Inundation depth (single output)  
    **start_water_level**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Start water level for flooding simulation  
    **end_water_level**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;End water level for flooding simulation  
    **water_level_step**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Step increment for water level in flooding simulation  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **memory**=*memory in MB*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Cache size for raster rows  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **-t**  
    &nbsp;&nbsp;&nbsp;&nbsp;Generate inundation raster maps for a series of water levels  
    **-m**  
    &nbsp;&nbsp;&nbsp;&nbsp;Use memory swap (operation is slow)  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **elevation** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input elevation raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **streams** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Stream raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the stream raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **direction** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Flow direction raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the flow direction raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **inundation_raster** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output inundation raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **hand** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Height above nearest drainage raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output HAND raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **inundation_strds** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output inundation STRDS.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, strds, *name*  
    **threshold** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basin threshold value  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *10000*  
    **depth** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Inundation depth (single output)  
    **start_water_level** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Start water level for flooding simulation  
    **end_water_level** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;End water level for flooding simulation  
    **water_level_step** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Step increment for water level in flooding simulation  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Cache size for raster rows  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *memory in MB*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *t*, *m*  
    &nbsp;&nbsp;&nbsp;&nbsp;**t**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Generate inundation raster maps for a series of water levels  
    &nbsp;&nbsp;&nbsp;&nbsp;**m**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use memory swap (operation is slow)  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **elevation** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input elevation raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **streams** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Stream raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the stream raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **direction** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Flow direction raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the flow direction raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **inundation_raster** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output inundation raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **hand** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Height above nearest drainage raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output HAND raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster, *name*  
    **inundation_strds** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of the output inundation STRDS.  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, strds, *name*  
    **threshold** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Basin threshold value  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *10000*  
    **depth** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Inundation depth (single output)  
    **start_water_level** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Start water level for flooding simulation  
    **end_water_level** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;End water level for flooding simulation  
    **water_level_step** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Step increment for water level in flooding simulation  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **memory** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Maximum memory to be used (in MB)  
    &nbsp;&nbsp;&nbsp;&nbsp;Cache size for raster rows  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: *memory in MB*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *300*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *t*, *m*  
    &nbsp;&nbsp;&nbsp;&nbsp;**t**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Generate inundation raster maps for a series of water levels  
    &nbsp;&nbsp;&nbsp;&nbsp;**m**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use memory swap (operation is slow)  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

**r.hand** - Height Above Nearest Drainage (HAND) is a terrain-based model that
computes the height of the terrain above the nearest stream using a digital
elevation model (DEM) [Nobre et al., 2011](https://doi.org/10.1016/j.jhydrol.2011.03.051).
The HAND method is widely used for hydrological analysis, particularly for estimating
flood inundation extents based on user-defined water levels.

The tool generates flood inundation extent maps by applying the HAND method to a
DEM. Users can optionally provide streams and flow direction raster maps to
improve processing efficiency and accuracy. The tool will return an inundation
raster map or time series (STRDS) of inundation raster maps based on user-defined
water levels. Additionally, the HAND raster map can be returned as an output if
desired by setting the *difference* parameter.

## Dependencies

### GRASS Addons

- [r.stream.distance](addons/r.stream.distance.md)
- [r.lake.series](addons/r.lake.series.md)

## EXAMPLES

### Calculate an inundation raster map from a DEM raster map

```bash
r.hand elevation=elevation hand=hand depth=2 inundation_raster=inundation
```

![Inundation event 2 m](r_hand.png)  
*Figure: Inundation event $2 m$.*

### Calculate series of inundation raster maps from a DEM raster map

```bash
r.hand -t elevation=elevation hand=hand inundation_strds=inundation_strds \
start_water_level=0 end_water_level=5 water_level_step=1
```

### Output HAND raster in addition to inundation raster(s)

```bash
r.hand elevation=elevation hand=hand depth=4 threshold=5000 \
inundation_raster=inundation
```

The HAND raster is classified according to Nobre et al (2011) where:

| HAND Range       | Classification        |
|------------------|-----------------------|
| 0 < HAND < 5 m   | Surface Water Table   |
| 5 < HAND < 15 m  | Shallow Water Table   |
| HAND > 15 m      | Deep Water Table      |

![r.hand HAND example](r_hand_color.png)
*Figure: Height Above Nearest Drainage (HAND)*

```bash
# Reclassify HAND raster into categories
r.reclass input=hand output=hand_reclass rules=- << EOF
-30000 thru 0 = NULL
1 thru 5 = 1 Surface
5 thru 15 = 2 Shallow
15 thru 30000 = 3 Deep
EOF

# Set color table for HAND raster
r.colors map=hand_reclass rules=- << EOF
1 #1d91c0
2 #41ab5d
3 #ec7014
nv white
default grey
EOF
```

![r.hand HAND classification example](r_hand_class.png)
*Figure: Height Above Nearest Drainage (HAND) water table classification.*

## REFERENCES

- Nobre, A.D., Cuartas, L.A., Hodnett, M., Rennó, C.D., Rodrigues, G., Silveira,
 A., Waterloo, M., Saleska, S., 2011. Height Above the Nearest Drainage – a
hydrologically relevant new terrain model. Journal of Hydrology 404, 13–29. <https://doi.org/10.1016/j.jhydrol.2011.03.051>

## SEE ALSO

*[r.watershed](r.watershed.md), [r.lake](r.lake.md), [r.lake.series](addons/r.lake.series.md)*

## AUTHORS

Corey T. White [OpenPlains Inc.](https://openplains.com)

## SOURCE CODE

Available at: [r.hand source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/raster/r.hand)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/raster/r.hand))  
Latest change: Tuesday Mar 04 09:58:40 2025 in commit [66645d6](https://github.com/OSGeo/grass-addons/commit/66645d69a3b663f079a42fb1309c25f8125f538c)
