---
name: r.roughness.vector.py
description: Calculates surface roughness in a moving-window, as the orientation of vectors normal to surface planes.
keywords: [ raster, terrain, aspect, slope, roughness ]
---

# r.roughness.vector.py

Calculates surface roughness in a moving-window, as the orientation of vectors normal to surface planes.

=== "Command line"

    **r.roughness.vector.py**
    **elevation**=*name*
    **slope**=*string*
    **aspect**=*string*
    [**window**=*integer*]
    [**strength**=*string*]
    [**fisher**=*string*]
    [**compass**=*string*]
    [**colatitude**=*string*]
    [**xcos**=*string*]
    [**ycos**=*string*]
    [**zcos**=*string*]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.roughness.vector.py elevation=name slope=string aspect=string
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.roughness.vector.py***",
        **elevation**,
        **slope**,
        **aspect**,
        **window**=*3*,
        **strength**=*None*,
        **fisher**=*None*,
        **compass**=*None*,
        **colatitude**=*None*,
        **xcos**=*None*,
        **ycos**=*None*,
        **zcos**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.roughness.vector.py", elevation="name", slope="string", aspect="string")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_roughness_vector_py*(**elevation**,
        **slope**,
        **aspect**,
        **window**=*3*,
        **strength**=*None*,
        **fisher**=*None*,
        **compass**=*None*,
        **colatitude**=*None*,
        **xcos**=*None*,
        **ycos**=*None*,
        **zcos**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_roughness_vector_py(elevation="name", slope="string", aspect="string")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **elevation**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of elevation raster map  
    **slope**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Input slope map  
    **aspect**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Input aspect map  
    **window**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Moving-window size (uses r.neighbors)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *3*  
    **strength**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output "vector strength" map  
    **fisher**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output "Fisher's K parameter" map  
    **compass**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input compass aspect map (optional)  
    **colatitude**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input colatitude map (optional)  
    **xcos**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input x directional cosine map (optional)  
    **ycos**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input y directional cosine map (optional)  
    **zcos**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input z directional cosine map (optional)  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **elevation** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of elevation raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **slope** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input slope map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **aspect** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input aspect map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **window** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Moving-window size (uses r.neighbors)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *3*  
    **strength** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output "vector strength" map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **fisher** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output "Fisher's K parameter" map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **compass** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input compass aspect map (optional)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **colatitude** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input colatitude map (optional)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **xcos** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input x directional cosine map (optional)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **ycos** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input y directional cosine map (optional)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **zcos** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input z directional cosine map (optional)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **elevation** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of elevation raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster, *name*  
    **slope** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input slope map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **aspect** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input aspect map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **window** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Moving-window size (uses r.neighbors)  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *3*  
    **strength** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output "vector strength" map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **fisher** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output "Fisher's K parameter" map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **compass** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input compass aspect map (optional)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **colatitude** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input colatitude map (optional)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **xcos** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input x directional cosine map (optional)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **ycos** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input y directional cosine map (optional)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **zcos** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Input z directional cosine map (optional)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

In this script surface roughness is taken as the dispersion of vectors
normal to surface areas (pixels). Normal vectors are defined by slope
and aspect.

This script will create several temporary maps, for the directional
cosines in each direction (x,y,z), for the sum of these cosines and
vector strength.

The options *compass*, *colatitude*, *xcos*, *ycosm* and *zcos* are
created as temporary files each time the script is run. If the user
wants to create several map (with different window sizes, for instance),
it is recommended to create those maps with *r.mapcalc* and use them as
input:

```sh
  r.mapcalc compass = "if(aspect==0,0,if(aspect < 90, 90-aspect, 360+90-aspect))"
  r.mapcalc colatitude = "90 - slope"
  r.mapcalc xcos = "sin(colatitude)*cos(compass)"
  r.mapcalc ycos = "sin(colatitude)*sin(compass)"
  r.mapcalc zcos = "cos(colatitude)"
 
```

If the user does not specify the output maps names, they will be set to

```text
INPUT_MAP_vector_strength_NxN
```

and

```text
INPUT_MAP_fisher_K_NxN
```

where N is the window size.

## EXAMPLE

```sh
  # calculate roughness factor by search window = 5
  r.roughness.vector elevation=DEM slope=slope aspect=aspect window=5
```

## SEE ALSO

*[r.mapcalc](https://grass.osgeo.org/grass-stable/manuals/r.mapcalc.html),
[r.slope.aspect](https://grass.osgeo.org/grass-stable/manuals/r.slope.aspect.html)*

## REFERENCES

Hobson, R.D., 1972. Surface roughness in topography: quantitative
approach. In: Chorley, R.J. (ed). *Spatial analysis in geomorphology*.
Methuer, London, p.225-245.  
  
McKean, J. & Roering, J., 2004. Objective landslide detection and
surface morphology mapping using high-resolution airborne laser
altimetry. *Geomorphology*, 57:331-351.
<https://doi.org/10.1016/S0169-555X(03)00164-8>.  
  
Grohmann, C.H., Smith, M.J. & Riccomini, C., 2011. Multiscale Analysis
of Topographic Surface Roughness in the Midland Valley, Scotland.
*Geoscience and Remote Sensing, IEEE Transactions on*, 49:1200-1213.
<https://doi.org/10.1109/TGRS.2010.2053546>

## AUTHORS

Carlos Henrique Grohmann - Institute of Energy and Environment,
University of São Paulo, Brazil. (<http://carlosgrohmann.com>)  
Helmut Kudrnovsky

## SOURCE CODE

Available at: [r.roughness.vector source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/raster/r.roughness.vector)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/raster/r.roughness.vector))  
Latest change: Thursday Mar 20 21:36:57 2025 in commit [7286ecf](https://github.com/OSGeo/grass-addons/commit/7286ecf7af235bfd089fb9b1b82fb383cf95f3fc)
