---
name: r.series.filter.py
description: Performs filtering of raster time series X (in time domain).
keywords: [ raster, statistics, filter ]
---

# r.series.filter.py

Performs filtering of raster time series X (in time domain).

=== "Command line"

    **r.series.filter.py**
    [**-cu**]
    **input**=*string* [,*string*,...]
    **result_prefix**=*string*
    [**method**=*string*]
    [**winsize**=*integer*]
    [**order**=*integer*]
    [**opt_points**=*integer*]
    [**diff_penalty**=*float*]
    [**deriv_penalty**=*float*]
    [**iterations**=*integer*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    r.series.filter.py input=string result_prefix=string
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***r.series.filter.py***",
        **input**,
        **result_prefix**,
        **method**=*"savgol"*,
        **winsize**=*9*,
        **order**=*2*,
        **opt_points**=*50*,
        **diff_penalty**=*1.0*,
        **deriv_penalty**=*1.0*,
        **iterations**=*1*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("r.series.filter.py", input="string", result_prefix="string")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.r_series_filter_py*(**input**,
        **result_prefix**,
        **method**=*"savgol"*,
        **winsize**=*9*,
        **order**=*2*,
        **opt_points**=*50*,
        **diff_penalty**=*1.0*,
        **deriv_penalty**=*1.0*,
        **iterations**=*1*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.r_series_filter_py(input="string", result_prefix="string")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*string* [,*string*,...] **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster names of equally spaced time series  
    **result_prefix**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Prefix for raster names of filtered X(t)  
    **method**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Used method  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *savgol*  
    **winsize**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Length of running window for the filter  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *9*  
    **order**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Order of the Savitzky-Golay filter  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2*  
    **opt_points**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Count of random points used for parameter optimization  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *50*  
    **diff_penalty**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Penalty for difference between original and filtered signals  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.0*  
    **deriv_penalty**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Penalty for big derivates of the filtered signal  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.0*  
    **iterations**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of iterations  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **-c**  
    &nbsp;&nbsp;&nbsp;&nbsp;Try to find optimal parameters for filtering  
    **-u**  
    &nbsp;&nbsp;&nbsp;&nbsp;Fit the result curve by upper boundary  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster names of equally spaced time series  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as:   
    **result_prefix** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Prefix for raster names of filtered X(t)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: q  
    **method** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Used method  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *savgol*  
    **winsize** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Length of running window for the filter  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *9*  
    **order** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Order of the Savitzky-Golay filter  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2*  
    **opt_points** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Count of random points used for parameter optimization  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *50*  
    **diff_penalty** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Penalty for difference between original and filtered signals  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.0*  
    **deriv_penalty** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Penalty for big derivates of the filtered signal  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.0*  
    **iterations** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of iterations  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *c*, *u*  
    &nbsp;&nbsp;&nbsp;&nbsp;**c**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Try to find optimal parameters for filtering  
    &nbsp;&nbsp;&nbsp;&nbsp;**u**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Fit the result curve by upper boundary  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str | list[str], *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Raster names of equally spaced time series  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as:   
    **result_prefix** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Prefix for raster names of filtered X(t)  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: q  
    **method** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Used method  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *savgol*  
    **winsize** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Length of running window for the filter  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *9*  
    **order** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Order of the Savitzky-Golay filter  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2*  
    **opt_points** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Count of random points used for parameter optimization  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *50*  
    **diff_penalty** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Penalty for difference between original and filtered signals  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.0*  
    **deriv_penalty** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Penalty for big derivates of the filtered signal  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1.0*  
    **iterations** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of iterations  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *c*, *u*  
    &nbsp;&nbsp;&nbsp;&nbsp;**c**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Try to find optimal parameters for filtering  
    &nbsp;&nbsp;&nbsp;&nbsp;**u**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Fit the result curve by upper boundary  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*r.series.filter* is a module to filter raster time series *X* in time
domain. It requires *python-scipy* (version 0.14 or later).

*-c*: Find optimal parameters of used filter. The function to optimize
depends on difference between original and filtered signals and on
derivates of the filtered signal.

*-u*: Filter using upper boundary of the signal values (Useful for
vegetation indexes filtering).

*input*: Raster names of equally spaced time series *X*.

*result\_prefix*: Prefix for raster names of filterd *X*.

*method*: Filtering method. Implemented filters are Savitzky-Golay
filter *savgol* and median filter *median*.

*winsize*: The length of the filter window. *winsize* must be a positive
odd integer.

*order*: The order of the polynomial used to fit the samples. The
*order* must be less than *winsize* (Savitzky-Golay only).

*iterations*: Number of filtering iterations.

*opt\_points*: If *-c* is specifed, then random sample *opt\_points* and
use them in parameter optimization.

*diff\_penalty*: Penalty for difference between original and filtered
signals (see Notes).

*deriv\_penalty*: Penalty for derivates of filtered signal (see Notes).

## NOTES

*X* must be equally spaced time series. If the series isn't equally
spaced, insert NULL raster maps into *X*.

There is a procedure for searching for good filtering parameters: it
uses *opt\_points* random points and perfoms filtering in that points.
The result of the filtering can be tested for quality. The quality
function is a trade of two features: accuracy and smoothing. Accuracy
can be estimated as the (abs) difference between original and filtered
data, quality of smoothing can be estimated as absalute values of the
derivates. So there are two parameters *diff\_penalty* and
*deriv\_penalty* that can ajust the trade-of.

So the optimizing procedure performs loop over filtering parameters and
calculates the next penalty function:

```text
penalty = diff_penalty * sum(abs(Xi-Fi)) + sum(abs(dFi))
```

where *Xi* are original signals in the samplig points, *Fi* are filtered
signals in the sampling points.

The optimal parameters are used for signal filtering in the whole
region.

If *-u* flag is specifed, then filter uses Chen's algorithm (see link
bellow). The algorithm is usefull for vegetation indexes filtering. It
creates a curve that flows on upper boundary of the signal.

## EXAMPLES

Create test data: *X = sin(t) + E*, where *X* is raster time series, *E*
is a error term.

```sh
for T in $(seq -w 0 10 360)
do
  name="test_raster"$T
  r.mapcalc -s "$name = sin($T) + rand(-0.3, 0.3)"
done
```

Create smooth raster series using Savitzky-Golay method:

```sh
maps=$(g.list rast patt="test_*" sep=,)
r.series.filter input=$maps result_prefix="flt." method=savgol winsize=9 order=2 --o
```

Look at the result (plot the curves for a pixel):

```sh
maps=$(g.list rast patt="test_*" sep=,)
fmaps=$(g.list rast patt="flt.*" sep=,)

eval $(g.region -cg)
i.spectral -g raster=$maps coor=$center_easting,$center_northing out=signal.png
i.spectral -g raster=$fmaps coor=$center_easting,$center_northing out=flt.png
```

![image-alt](signal.png) ![image-alt](flt.png)

## REFERENCES

Chen, Jin; Jonsson, Per; Tamura, Masayuki; Gu, Zhihui; Matsushita,
Bunkei; Eklundh, Lars. (2004). *A simple method for reconstructing a
high-quality NDVI time-series data set based on the Savitzky-Golay
filter*. **Remote Sensing of Environment**, 91, 332-344,
doi:[10.1016/j.rse.2004.03.014](https://doi.org/10.1016/j.rse.2004.03.014).

## SEE ALSO

*[r.regression.series](r.regression.series.md),
[r.series](https://grass.osgeo.org/grass-stable/manuals/r.series.html),
[r.regression.line](https://grass.osgeo.org/grass-stable/manuals/r.regression.line.html),
[g.list](https://grass.osgeo.org/grass-stable/manuals/g.list.html)*

## AUTHOR

Dmitry Kolesov

## SOURCE CODE

Available at: [r.series.filter source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/raster/r.series.filter)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/raster/r.series.filter))  
Latest change: Thursday Mar 20 21:36:57 2025 in commit [7286ecf](https://github.com/OSGeo/grass-addons/commit/7286ecf7af235bfd089fb9b1b82fb383cf95f3fc)
