---
name: v.stream.network.py
description: Build a linked stream network: each link knows its downstream link
keywords: [ vector, stream network, hydrology, geomorphology ]
---

# v.stream.network.py

Build a linked stream network: each link knows its downstream link

=== "Command line"

    **v.stream.network.py**
    **map**=*name*
    [**upstream_easting_column**=*string*]
    [**upstream_northing_column**=*string*]
    [**downstream_easting_column**=*string*]
    [**downstream_northing_column**=*string*]
    [**tostream_cat_column**=*string*]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    v.stream.network.py map=name
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***v.stream.network.py***",
        **map**,
        **upstream_easting_column**=*"x1"*,
        **upstream_northing_column**=*"y1"*,
        **downstream_easting_column**=*"x2"*,
        **downstream_northing_column**=*"y2"*,
        **tostream_cat_column**=*"tostream"*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("v.stream.network.py", map="name")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.v_stream_network_py*(**map**,
        **upstream_easting_column**=*"x1"*,
        **upstream_northing_column**=*"y1"*,
        **downstream_easting_column**=*"x2"*,
        **downstream_northing_column**=*"y2"*,
        **tostream_cat_column**=*"tostream"*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.v_stream_network_py(map="name")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **map**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector stream network from r.stream.extract  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    **upstream_easting_column**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Upstream easting (or x or lon) column name  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *x1*  
    **upstream_northing_column**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Upstream northing (or y or lat) column name  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *y1*  
    **downstream_easting_column**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Downstream easting (or x or lon) column name  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *x2*  
    **downstream_northing_column**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Downstream northing (or y or lat) column name  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *y2*  
    **tostream_cat_column**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Adjacent downstream segment category  
    &nbsp;&nbsp;&nbsp;&nbsp;Zero (0) indicates off-map flow  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *tostream*  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **map** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector stream network from r.stream.extract  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **upstream_easting_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Upstream easting (or x or lon) column name  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *x1*  
    **upstream_northing_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Upstream northing (or y or lat) column name  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *y1*  
    **downstream_easting_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Downstream easting (or x or lon) column name  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *x2*  
    **downstream_northing_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Downstream northing (or y or lat) column name  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *y2*  
    **tostream_cat_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Adjacent downstream segment category  
    &nbsp;&nbsp;&nbsp;&nbsp;Zero (0) indicates off-map flow  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *tostream*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **map** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Vector stream network from r.stream.extract  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **upstream_easting_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Upstream easting (or x or lon) column name  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *x1*  
    **upstream_northing_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Upstream northing (or y or lat) column name  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *y1*  
    **downstream_easting_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Downstream easting (or x or lon) column name  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *x2*  
    **downstream_northing_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Downstream northing (or y or lat) column name  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *y2*  
    **tostream_cat_column** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Adjacent downstream segment category  
    &nbsp;&nbsp;&nbsp;&nbsp;Zero (0) indicates off-map flow  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *tostream*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*v.stream.network* populates the attribute table of a river network
built by
[r.stream.extract](https://grass.osgeo.org/grass-stable/manuals/r.stream.extract.html)
with five new columns that define the nodes and links within a river
network. These are (default names given):

- *x1*, *y1*: upstream coordinates
- *x2*, *y2*: downstream coordinates
- *tostream*: category number of the next stream; is equal to 0 if the
    stream flows off of the map

Any stream-like network will work, but the lines need to be connected.
In terms of graph theory, a tree and forest are supported. Behavior for
a cyclic graph is undefined.

## NOTES

**streams** is a set of vector lines that is generated by
r.stream.extract. It is recommended to be built as follows (Python
code):

```sh
# Choose elevation map and threshold
elevation = srtm
threshold = 1E6 # [m**2], minimum catchment size = 1 km**2
# Build a map of cell areas: pick one of these
# km2 is more intuitive, but 1 km2 is the threshold for the smallest
# threshold catchment size
r.mapcalc('cellArea_meters2 = '+str(reg.nsres)+' * '+str(reg.ewres), overwrite=True)
r.mapcalc("cellArea_km2 = cellArea_meters2 / 10^6", overwrite=True)

# Use r.watershed to build a drainage basin with units of length**2
# instead of arbitrary units (cells)
r.watershed(elevation=elevation, flow='cellArea_meters2', accumulation='drainageArea_m2', drainage='drainageDirection', stream='streams', threshold=thresh, flags='s', overwrite=True)
# Note that this will include areas of negative (i.e. offmap) flow accumulation

# Build watershed network using r.stream.extract: single-flow-direction (SFD)
print "Building drainage network"
r.stream_extract(elevation=elevation, accumulation='drainageArea_m2', threshold=thresh, d8cut=0, mexp=0, stream_raster='streams', stream_vector='streams', direction='draindir', overwrite=True)
```

## REFERENCES

- Ng, G-H. Crystal, Andrew D. Wickert, Lauren D. Somers, Leila Saberi,
    Collin Cronkite-Ratcliff, Richard G. Niswonger, and Jeffrey M.
    McKenzie. "GSFLOW–GRASS v1. 0.0: GIS-enabled hydrologic modeling of
    coupled groundwater–surface-water systems." *Geoscientific Model
    Development* 11 (2018): 4755-4777.
    [DOI 10.5194/gmd-11-4755-2018](https://doi.org/10.5194/gmd-11-4755-2018)

## SEE ALSO

*[r.stream.extract](https://grass.osgeo.org/grass-stable/manuals/r.stream.extract.html),
[v.stream.order](v.stream.order.md)*

## AUTHOR

Andrew D. Wickert

## SOURCE CODE

Available at: [v.stream.network source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/vector/v.stream.network)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/vector/v.stream.network))  
Latest change: Friday Feb 21 10:10:05 2025 in commit [7d78fe3](https://github.com/OSGeo/grass-addons/commit/7d78fe34868674c3b6050ba1924e1c5675d155c9)
