---
name: v.surf.icw.py
description: IDW interpolation, but distance is cost to get to any other site.
keywords: [ vector, surface, interpolation, ICW ]
---

# v.surf.icw.py

IDW interpolation, but distance is cost to get to any other site.

=== "Command line"

    **v.surf.icw.py**
    [**-r**]
    **input**=*string*
    **column**=*string*
    **output**=*string*
    **cost_map**=*string*
    [**friction**=*float*]
    [**layer**=*integer*]
    [**where**=*string*]
    [**post_mask**=*string*]
    [**workers**=*integer*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    v.surf.icw.py input=string column=string output=string cost_map=string
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***v.surf.icw.py***",
        **input**,
        **column**,
        **output**,
        **cost_map**,
        **friction**=*2*,
        **layer**=*1*,
        **where**=*None*,
        **post_mask**=*None*,
        **workers**=*1*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("v.surf.icw.py", input="string", column="string", output="string", cost_map="string")
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.v_surf_icw_py*(**input**,
        **column**,
        **output**,
        **cost_map**,
        **friction**=*2*,
        **layer**=*1*,
        **where**=*None*,
        **post_mask**=*None*,
        **workers**=*1*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.v_surf_icw_py(input="string", column="string", output="string", cost_map="string")
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of existing vector points map containing seed data  
    **column**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Column name in points map that contains data values  
    **output**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    **cost_map**=*string* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of existing raster map containing cost information  
    **friction**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Friction of distance, (the 'n' in 1/d^n)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-6*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2*  
    **layer**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number of data in points map  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **where**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL query statement without 'where' keyword  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: income &lt; 1000 and inhab &gt;= 10000  
    **post_mask**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of existing raster map to be used as post-processing MASK  
    **workers**=*integer*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of parallel processes to launch  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-256*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **-r**  
    &nbsp;&nbsp;&nbsp;&nbsp;Use (d^n)\*log(d) instead of 1/(d^n) for radial basis function  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of existing vector points map containing seed data  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector  
    **column** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Column name in points map that contains data values  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster  
    **cost_map** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of existing raster map containing cost information  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **friction** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Friction of distance, (the 'n' in 1/d^n)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-6*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2*  
    **layer** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number of data in points map  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **where** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL query statement without 'where' keyword  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: income &lt; 1000 and inhab &gt;= 10000  
    **post_mask** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of existing raster map to be used as post-processing MASK  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **workers** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of parallel processes to launch  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-256*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *r*  
    &nbsp;&nbsp;&nbsp;&nbsp;**r**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use (d^n)\*log(d) instead of 1/(d^n) for radial basis function  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of existing vector points map containing seed data  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector  
    **column** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Column name in points map that contains data values  
    **output** : str | type(np.ndarray) | type(np.array) | type(gs.array.array), *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output raster map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, raster  
    **cost_map** : str | np.ndarray, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of existing raster map containing cost information  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **friction** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Friction of distance, (the 'n' in 1/d^n)  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-6*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *2*  
    **layer** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Layer number of data in points map  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **where** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;WHERE conditions of SQL query statement without 'where' keyword  
    &nbsp;&nbsp;&nbsp;&nbsp;Example: income &lt; 1000 and inhab &gt;= 10000  
    **post_mask** : str | np.ndarray, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of existing raster map to be used as post-processing MASK  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, raster  
    **workers** : int, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Number of parallel processes to launch  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *1-256*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *1*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *r*  
    &nbsp;&nbsp;&nbsp;&nbsp;**r**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use (d^n)\*log(d) instead of 1/(d^n) for radial basis function  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | np.ndarray | tuple[np.ndarray] | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned. If an array type (e.g., *np.ndarray*) is used for one of the raster outputs, the result will be an array and will have the shape corresponding to the computational region. If an array type is used for more than one raster output, the result will be a tuple of arrays.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

Inverse cost weighting is like inverse distance weighted (IDW)
interpolation, but uses *cost* instead of shortest Euclidean distance.
In this way solid barriers and molasses zones may be correctly taken
into account.

Input data points do not need to have direct line of sight to each
other. This is interpolation "as the fish swims", not "as the crow
flies", and can see around headlands or across land-bridges without
polluting over barriers which the natural value (or flightless bird) can
not cross.

It was initially written to interpolate water chemistry in two parallel
arms of a fjord, but may just as well be used for population abundance
constrained by topography, or in studies of archeologic technology
transfer.

## NOTES

In the simplest case, the cost map will just be a mask raster with
values of 1 in areas to interpolate, and NULL in impenetrable areas.
Fancier cost maps can be used, for example, to make it more expensive
for a measured pollutant to diffuse upstream in an estuary, or to make
it more expensive for a stone tool technology to cross waterways.

Since generating cost maps can take a long time, it is recommended to
keep the raster region relatively small and limit the number of starting
points to less than 500.

Higher values of **friction** will help limit unconstrained boundary
effects at the edges of your coverage, but will incur more of a stepped
transition between sites.

The **post\_mask**, if given, is applied after the interpolation is
complete. A common use for that might be to only present data within a
certain distance (thus confidence) of an actual sampling station. In
that case the *r.cost* module can be used to create the mask.

This module writes lots of temporary files and so can be rather disk I/O
intensive. If you are running it many times in a big loop you may want
to try setting up a RAM-disk to put the mapset in (on UNIX symlinking
back into the location is ok), or adjust the disk-cache flushing timer
to be slightly longer than one iteration of the script.  
To do this on Linux you can tune the
`/proc/sys/vm/dirty_expire_centisecs` kernel control. The default is to
keep files in memory a maximum of 30 seconds before writing them to
disk, although if you are short on free RAM the kernel may write to disk
long before that timeout is reached.

By default the module will run serially. To run in parallel set the
**workers** parameter to the desired value (typically the number of
cores in your CPU). Alternatively, if the `WORKERS` environment variable
is set, the number of concurrent processes will be set at that number of
jobs.

## EXAMPLE

In this example we'll generate some fake island barriers from the
standard North Carolina GRASS dataset, then interpolate a continuous
variable between given point stations. We'll use rainfall, but for the
purposes of the exercise pretend it is a nutrient concentration in our
fake wayerway. Point data stations outside of the current region or in
NULL areas of the *cost\_map* will be ignored.

```sh
# set up a fake sea with islands:
g.region n=276000 s=144500 w=122000 e=338500 res=500
r.mapcalc "pseudo_elev = elev_state_500m - 1100"
r.colors pseudo_elev color=etopo2
r.mapcalc "navigable_mask = if(pseudo_elev < 0, 1, null())"

# pick a data column from the points vector:
v.info -c precip_30ynormals

# run the interpolation:
v.surf.icw input=precip_30ynormals column=annual output=annual_interp.3 \
   cost_map=navigable_mask friction=3 --verbose

# equalize colors to show maximum detail:
r.colors -e annual_interp.3 color=bcyr

# display results in a GRASS monitor:
d.mon wx0
d.erase black
d.rast annual_interp.3
d.vect precip_30ynormals fcolor=red icon=basic/circle
d.legend annual_interp.3 at=48.4,94.8,3.4,6.0
```

## REFERENCES

The method was first described in Wing et. al 2004, with further
comments and examples in report 3 of that series, 2005. Ducke and
Rassmann 2010 (in German) describe a novel use of the approach to study
prehistoric movement corridors of early Bronze Age technology through
Europe.

- Wing, S.R., M.H.E. Bowman, F. Smith and S.M. Rutger (2004),
    *Analysis of biodiversity patterns and management decision making
    processes to support stewardship of marine resources and
    biodiversity in Fiordland - a case study*, report 2 of 3, Technical
    report, Report to the Ministry for the Environment, New Zealand.
- Ducke, B. and K. Rassmann (2010), *Modelling and Interpretation of
    the Communication Spaces of the 3rd and Early 2nd Millennium BC in
    Europe Using Diversity Gradients*   (*Modellierung und
    Interpretation der Kommunikationsräume des 3. und frühen 2.
    Jahrtausends v. Chr.*), in Europa mittels Diversitätsgradienten;
    Archäologischer Anzeiger 2010/1, pp.239-261

## SEE ALSO

*[v.surf.idw](https://grass.osgeo.org/grass-stable/manuals/v.surf.idw.html).
[v.surf.rst](https://grass.osgeo.org/grass-stable/manuals/v.surf.rst.html).
[v.surf.bspline](https://grass.osgeo.org/grass-stable/manuals/v.surf.bspline.html).
[r.cost](https://grass.osgeo.org/grass-stable/manuals/r.cost.html).
[r.surf.idw](https://grass.osgeo.org/grass-stable/manuals/r.surf.idw.html),
[r.surf.idw2](https://grass.osgeo.org/grass-stable/manuals/r.surf.idw2.html)*

## AUTHOR

Hamish Bowman  
*Department of Marine Science,  
Dunedin, New Zealand*

## SOURCE CODE

Available at: [v.surf.icw source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/vector/v.surf.icw)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/vector/v.surf.icw))  
Latest change: Friday Feb 21 10:10:05 2025 in commit [7d78fe3](https://github.com/OSGeo/grass-addons/commit/7d78fe34868674c3b6050ba1924e1c5675d155c9)
