---
name: v.transects.py
description: Creates transect lines or quadrilateral areas at regular intervals perpendicular to a polyline.
keywords: [ vector, transect ]
---

# v.transects.py

Creates transect lines or quadrilateral areas at regular intervals perpendicular to a polyline.

=== "Command line"

    **v.transects.py**
    [**-l**]
    **input**=*name*
    **output**=*name*
    **transect_spacing**=*float*
    [**dleft**=*float*]
    [**dright**=*float*]
    [**type**=*string*]
    [**metric**=*string*]
    [**transect_perpendicular**=*string*]
    [**--overwrite**]
    [**--verbose**]
    [**--quiet**]
    [**--qq**]
    [**--ui**]

    Example:

    ```sh
    v.transects.py input=name output=name transect_spacing=0.0
    ```

=== "Python (grass.script)"

    *grass.script.run_command*("***v.transects.py***",
        **input**,
        **output**,
        **transect_spacing**,
        **dleft**=*None*,
        **dright**=*None*,
        **type**=*"line"*,
        **metric**=*"straight"*,
        **transect_perpendicular**=*"trend"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    gs.run_command("v.transects.py", input="name", output="name", transect_spacing=0.0)
    ```

=== "Python (grass.tools)"

    *grass.tools.Tools.v_transects_py*(**input**,
        **output**,
        **transect_spacing**,
        **dleft**=*None*,
        **dright**=*None*,
        **type**=*"line"*,
        **metric**=*"straight"*,
        **transect_perpendicular**=*"trend"*,
        **flags**=*None*,
        **overwrite**=*None*,
        **verbose**=*None*,
        **quiet**=*None*,
        **superquiet**=*None*)

    Example:

    ```python
    tools = Tools()
    tools.v_transects_py(input="name", output="name", transect_spacing=0.0)
    ```

    This grass.tools API is experimental in version 8.5 and expected to be stable in version 8.6.

## Parameters

=== "Command line"

    **input**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    **output**=*name* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output vector map  
    **transect_spacing**=*float* **[required]**  
    &nbsp;&nbsp;&nbsp;&nbsp;Transect spacing along input polyline  
    **dleft**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Distance transect extends to the left of input line  
    &nbsp;&nbsp;&nbsp;&nbsp;Default is the same as the transect spacing  
    **dright**=*float*  
    &nbsp;&nbsp;&nbsp;&nbsp;Distance transect extends to the right of input line  
    &nbsp;&nbsp;&nbsp;&nbsp;Default is the same as the transect spacing  
    **type**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output type  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *point, line, area*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *line*  
    **metric**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Determines how transect spacing is measured  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *straight,  along*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *straight*  
    &nbsp;&nbsp;&nbsp;&nbsp;**straight**: Straight distance between transect points  
    &nbsp;&nbsp;&nbsp;&nbsp;**along**: Distance along the line  
    **transect_perpendicular**=*string*  
    &nbsp;&nbsp;&nbsp;&nbsp;Determines which line is the transect perpendicular to  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *trend,  line*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *trend*  
    &nbsp;&nbsp;&nbsp;&nbsp;**trend**: Perpendicular to the line connecting transect points  
    &nbsp;&nbsp;&nbsp;&nbsp;**line**: Perpendicular to the particular segment of the original line  
    **-l**  
    &nbsp;&nbsp;&nbsp;&nbsp;Use the last point of the line to create transect  
    **--overwrite**  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    **--help**  
    &nbsp;&nbsp;&nbsp;&nbsp;Print usage summary  
    **--verbose**  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    **--quiet**  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    **--qq**  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    **--ui**  
    &nbsp;&nbsp;&nbsp;&nbsp;Force launching GUI dialog

=== "Python (grass.script)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, vector, *name*  
    **transect_spacing** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Transect spacing along input polyline  
    **dleft** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Distance transect extends to the left of input line  
    &nbsp;&nbsp;&nbsp;&nbsp;Default is the same as the transect spacing  
    **dright** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Distance transect extends to the right of input line  
    &nbsp;&nbsp;&nbsp;&nbsp;Default is the same as the transect spacing  
    **type** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output type  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *point, line, area*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *line*  
    **metric** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Determines how transect spacing is measured  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *straight,  along*  
    &nbsp;&nbsp;&nbsp;&nbsp;**straight**: Straight distance between transect points  
    &nbsp;&nbsp;&nbsp;&nbsp;**along**: Distance along the line  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *straight*  
    **transect_perpendicular** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Determines which line is the transect perpendicular to  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *trend,  line*  
    &nbsp;&nbsp;&nbsp;&nbsp;**trend**: Perpendicular to the line connecting transect points  
    &nbsp;&nbsp;&nbsp;&nbsp;**line**: Perpendicular to the particular segment of the original line  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *trend*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *l*  
    &nbsp;&nbsp;&nbsp;&nbsp;**l**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use the last point of the line to create transect  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

=== "Python (grass.tools)"

    **input** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name of input vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Or data source for direct OGR access  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: input, vector, *name*  
    **output** : str, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Name for output vector map  
    &nbsp;&nbsp;&nbsp;&nbsp;Used as: output, vector, *name*  
    **transect_spacing** : float, *required*  
    &nbsp;&nbsp;&nbsp;&nbsp;Transect spacing along input polyline  
    **dleft** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Distance transect extends to the left of input line  
    &nbsp;&nbsp;&nbsp;&nbsp;Default is the same as the transect spacing  
    **dright** : float, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Distance transect extends to the right of input line  
    &nbsp;&nbsp;&nbsp;&nbsp;Default is the same as the transect spacing  
    **type** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Output type  
    &nbsp;&nbsp;&nbsp;&nbsp;Input feature type  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *point, line, area*  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *line*  
    **metric** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Determines how transect spacing is measured  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *straight,  along*  
    &nbsp;&nbsp;&nbsp;&nbsp;**straight**: Straight distance between transect points  
    &nbsp;&nbsp;&nbsp;&nbsp;**along**: Distance along the line  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *straight*  
    **transect_perpendicular** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Determines which line is the transect perpendicular to  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *trend,  line*  
    &nbsp;&nbsp;&nbsp;&nbsp;**trend**: Perpendicular to the line connecting transect points  
    &nbsp;&nbsp;&nbsp;&nbsp;**line**: Perpendicular to the particular segment of the original line  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *trend*  
    **flags** : str, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allowed values: *l*  
    &nbsp;&nbsp;&nbsp;&nbsp;**l**  
    &nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;Use the last point of the line to create transect  
    **overwrite** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Allow output files to overwrite existing files  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **verbose** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Verbose module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **quiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  
    **superquiet** : bool, *optional*  
    &nbsp;&nbsp;&nbsp;&nbsp;Very quiet module output  
    &nbsp;&nbsp;&nbsp;&nbsp;Default: *None*  

    Returns:

    **result** : grass.tools.support.ToolResult | None  
    If the tool produces text as standard output, a *ToolResult* object will be returned. Otherwise, `None` will be returned.

    Raises:

    *grass.tools.ToolError*: When the tool ended with an error.

## DESCRIPTION

*v.transects* creates equally spaced geometries along input lines.
Depending on the selected **metric**, the distances are measured either
as straight distances between the transects or along the line. Straight
distance is better when the lines are complicated (shorelines) and using
the distance along line would result in big differences in the transect
spacing.

The geometries can be lines or quadrilateral areas. Lines and areas are
generated to be perpendicular to either the input line (select *line*
for parameter **transect\_perpendicular**), or to the line connecting
the transect points (select *trend* for parameter
**transect\_perpendicular**). The latter option is useful for
complicated lines, because it follows the trend of the line.

By default the last point of each line is not used because the last
transect distance would be typically different from the previous once.
To create a transect from the last point, use flag **-l**.

The following pictures show the effect of different options on a line
with simple geometry. In the first picture, the circles show points
generated by
[v.segment](https://grass.osgeo.org/grass-stable/manuals/v.segment.html)
in 100 m interval along the line. The green lines were generated with
option `metric=along`, therefore they match with the points. The blue
lines were generated with `metric=straight`.  
In the second picture, green lines use `transect_perpendicular=line`
while the blue ones use `transect_perpendicular=trend`.

![image-alt](vtransect_options1.jpg)
![image-alt](vtransect_options2.jpg)

## NOTES

Input vector lines that are shorter than **transect\_spacing** are
ignored.

**transect\_spacing**, **dleft**, and **dright** are interpreted to be
in horizontal map units (e.g., degrees in the LatLong/WGS84 coordinate
system).

***v.transects*** may fail for a network of lines in Windows.

## EXAMPLES

In these examples, the [Nags Head
(19MB)](https://web.archive.org/web/20240225135759/https://fatra.cnr.ncsu.edu/geospatial-modeling-course/data/NagsHead_series.zip)
mapset is used to generate a shoreline and shore-perpendicular
geometries. To use the mapset, unpack it into the [nc\_spm\_08\_grass7
(50MB)](https://grass.osgeo.org/sampledata/north_carolina/nc_basic_spm_grass7.zip)
location.

### Example 1) - Generate line transects along shoreline

Generate 20 cross-shore transects along 2008 shoreline (1m contour)  

```sh
g.region raster=NH_2008_1m
r.contour input=NH_2008_1m output=NH_2008_1m level=1 cut=100
v.report map=NH_2008_1m option=length
# cat|level|length
# 1|1|1037.86684790028
# 1038m / 20transects = 52m per transect (value for transect_spacing)
v.transects input=NH_2008_1m output=NH_2008_transects transect_spacing=52
v.info NH_2008_transects
```

### Example 2) - Generate line transects specifying the left and right length

Generate longer, more parallel transects by specifying dleft and dright
and smoothing the input line  

```sh
g.region raster=NH_2008_1m
r.contour input=NH_2008_1m output=NH_2008_1m level=1 cut=100
v.generalize input=NH_2008_1m output=NH_2008_1m_smoothed \
  method=sliding_averaging look_ahead=201
v.transects input=NH_2008_1m_smoothed \
  output=NH_2008_transects_long_smoothed transect_spacing=52 \
  dleft=20 dright=300
```

### Example 3) - Generate area transects along shoreline

Generate longer, more parallel transects by specifying dleft and dright
and smoothing the input line  

```sh
g.region raster=NH_2008_1m
r.contour input=NH_2008_1m output=NH_2008_1m level=1 cut=100
v.transects input=NH_2008_1m output=NH_2008_areas \
  transect_spacing=52 dleft=20 dright=300 type=area
v.db.addtable NH_2008_areas
v.db.addcolumn map=NH_2008_areas columns='demStats DOUBLE PRECISION'
v.rast.stats vector=NH_2008_areas raster=NH_2008_1m column_prefix=NH2008
v.db.select NH_2008_areas
```

## SEE ALSO

[v.segment](https://grass.osgeo.org/grass-stable/manuals/v.segment.html)  
[v.generalize](https://grass.osgeo.org/grass-stable/manuals/v.generalize.html)  
[v.transects
tutorial](https://web.archive.org/web/20170904061203/https://www4.ncsu.edu/~ejhardi2/vTransect.html)

## AUTHORS

Eric Hardin, Helena Mitasova, Updates by John Lloyd

## SOURCE CODE

Available at: [v.transects source code](https://github.com/OSGeo/grass-addons/tree/grass8/src/vector/v.transects)
([history](https://github.com/OSGeo/grass-addons/commits/grass8/src/vector/v.transects))  
Latest change: Saturday Jul 19 20:24:44 2025 in commit [970afd7](https://github.com/OSGeo/grass-addons/commit/970afd728e4ec436cecd8604fd4b048f80417073)
